package org.palladiosimulator.indirections.scheduler.util;

import java.io.NotSerializableException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.log4j.Logger;
import org.eclipse.emf.common.util.EList;
import org.palladiosimulator.indirections.interfaces.IndirectionDate;
import org.palladiosimulator.indirections.repository.DataChannel;
import org.palladiosimulator.indirections.repository.DataInterface;
import org.palladiosimulator.indirections.repository.JavaClassDataChannel;
import org.palladiosimulator.indirections.scheduler.data.ConcreteGroupingIndirectionDate;
import org.palladiosimulator.indirections.scheduler.data.ConcreteIndirectionDate;
import org.palladiosimulator.indirections.scheduler.data.GenericJoinedDate;
import org.palladiosimulator.indirections.util.IterableUtil;
import org.palladiosimulator.pcm.core.PCMRandomVariable;
import org.palladiosimulator.pcm.core.entity.Entity;
import org.palladiosimulator.pcm.parameter.VariableCharacterisation;
import org.palladiosimulator.pcm.parameter.VariableUsage;
import org.palladiosimulator.pcm.repository.EventGroup;
import org.palladiosimulator.pcm.repository.Parameter;
import org.palladiosimulator.pcm.stoex.api.StoExSerialiser;
import org.palladiosimulator.simulizar.exceptions.PCMModelInterpreterException;
import org.palladiosimulator.simulizar.simulationevents.PeriodicallyTriggeredSimulationEntity;
import org.palladiosimulator.simulizar.utils.SimulatedStackHelper;

import de.uka.ipd.sdq.identifier.Identifier;
import de.uka.ipd.sdq.simucomframework.core.entities.SimuComEntity;
import de.uka.ipd.sdq.simucomframework.core.model.SimuComModel;
import de.uka.ipd.sdq.simucomframework.variables.EvaluationProxy;
import de.uka.ipd.sdq.simucomframework.variables.StackContext;
import de.uka.ipd.sdq.simucomframework.variables.exceptions.ValueNotInFrameException;
import de.uka.ipd.sdq.simucomframework.variables.stackframe.SimulatedStack;
import de.uka.ipd.sdq.simucomframework.variables.stackframe.SimulatedStackframe;
import de.uka.ipd.sdq.stoex.AbstractNamedReference;

public final class IndirectionSimulationUtil {
    public final static class Pair<A, B> {
        public final A a;
        public final B b;

        public Pair(A a, B b) {
            this.a = a;
            this.b = b;
        }

        @Override
        public int hashCode() {
            return Objects.hash(a, b);
        }

        @Override
        public boolean equals(Object obj) {
            if (this == obj)
                return true;
            if (obj == null)
                return false;
            if (getClass() != obj.getClass())
                return false;
            @SuppressWarnings("rawtypes")
            Pair other = (Pair) obj;
            return Objects.equals(a, other.a) && Objects.equals(b, other.b);
        }

        public static <A, B> Pair<A, B> of(A a, B b) {
            return new Pair<>(a, b);
        }
    }

    private final static Logger LOGGER = Logger.getLogger(IndirectionSimulationUtil.class);
    private final static StoExSerialiser STOEX_SERIALISER = StoExSerialiser.createInstance();

    /**
     * Same as
     * {@link SimulatedStackHelper#addParameterToStackFrame(SimulatedStackframe, EList, SimulatedStackframe)}
     * but defaults for the parameters.
     *
     * Additionally, it can copy all characterizations for a type by specifying a reference name of:
     * input->output
     *
     * @param parameterName
     */
    public static final void addParameterToStackFrameWithCopying(final SimulatedStackframe<Object> contextStackFrame,
            final EList<VariableUsage> parameter, final String parameterName,
            final SimulatedStackframe<Object> targetStackFrame) {

        for (final VariableUsage variableUsage : parameter) {
            if (variableUsage.getVariableCharacterisation_VariableUsage()
                .isEmpty()) {
                final AbstractNamedReference namedReference = variableUsage.getNamedReference__VariableUsage();
                // TODO: move from convention quick hack to better solution
                final String[] split = namedReference.getReferenceName()
                    .split("->");
                if (split.length != 2) {
                    throw new PCMModelInterpreterException(
                            "If no variable chacterisations are present, name must be of form 'input->output'. Name is: "
                                    + namedReference.getReferenceName());
                }

                final String inputPrefix = split[0] + ".";
                final String outputPrefix = split[1] + ".";

                final List<Entry<String, Object>> inputs = contextStackFrame.getContents()
                    .stream()
                    .filter(it -> it.getKey()
                        .startsWith(inputPrefix))
                    .collect(Collectors.toList());

                if (inputs.size() == 0) {
                    throw new PCMModelInterpreterException("Nothing found on stack frame for prefix '" + inputPrefix
                            + "'. Available: " + contextStackFrame.getContents()
                                .stream()
                                .map(it -> it.getKey())
                                .collect(Collectors.joining(", ")));
                }

                inputs.forEach(it -> targetStackFrame.addValue(outputPrefix + it.getKey()
                    .substring(inputPrefix.length()), it.getValue()));
                continue;
            }

            for (final VariableCharacterisation variableCharacterisation : variableUsage
                .getVariableCharacterisation_VariableUsage()) {

                final PCMRandomVariable randomVariable = variableCharacterisation
                    .getSpecification_VariableCharacterisation();

                final AbstractNamedReference namedReference = variableCharacterisation
                    .getVariableUsage_VariableCharacterisation()
                    .getNamedReference__VariableUsage();

                String id;
                try {
                    id = STOEX_SERIALISER.serialise(namedReference)
                        .toString() + "."
                            + variableCharacterisation.getType()
                                .getLiteral();
                } catch (NotSerializableException e1) {
                    throw new PCMModelInterpreterException("Could not serialise a named reference.", e1);
                }

                if (SimulatedStackHelper.isInnerReference(namedReference)) {
                    targetStackFrame.addValue(id,
                            new EvaluationProxy(randomVariable.getSpecification(), contextStackFrame.copyFrame()));
                } else {
                    targetStackFrame.addValue(id,
                            StackContext.evaluateStatic(randomVariable.getSpecification(), contextStackFrame));
                }

                if (LOGGER.isDebugEnabled()) {
                    try {
                        LOGGER.debug("Added value " + targetStackFrame.getValue(id) + " for id " + id
                                + " to stackframe " + targetStackFrame);
                    } catch (final ValueNotInFrameException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }
    }

    public static IndirectionDate claimDataFromStack(final SimulatedStack<Object> stack, final String id) {
        final SimulatedStackframe<Object> currentStackFrame = stack.currentStackFrame();
        Object value;
        try {
            value = currentStackFrame.getValue(id);
        } catch (final ValueNotInFrameException e) {
            throw new PCMModelInterpreterException("Expected id " + id + " on stack, but not found.", e);
        }
        if (!(value instanceof IndirectionDate)) {
            throw new PCMModelInterpreterException("Expected " + IndirectionDate.class.getName() + " for id " + id
                    + ", but got " + value.toString() + "(" + value.getClass()
                        .getName()
                    + ")");
        }
        return (IndirectionDate) value;
    }

    public static IndirectionDate createData(final SimulatedStack<Object> contextStack,
            final Iterable<VariableUsage> variableUsages, Collection<Double> time) {

        final Map<String, Object> entries = createDataEntries(contextStack, variableUsages);
        final IndirectionDate result = new ConcreteIndirectionDate(entries, time);

        return result;
    }

    public static Map<String, Object> createDataEntries(final SimulatedStack<Object> contextStack,
            final Iterable<VariableUsage> variableUsages) {
        final SimulatedStackframe<Object> newStackFrame = new SimulatedStackframe<Object>();
        SimulatedStackHelper.addParameterToStackFrame(contextStack.currentStackFrame(),
                IterableUtil.toEList(variableUsages), newStackFrame);

        final Map<String, Object> entries = IterableUtil.toMap(newStackFrame.getContents());

        return entries;
    }

    // TODO: it is not very good to do exception handling checks here. This should be functionality
    // of the StackFrame, if it is really needed. Additionally, it is unclear, whether this should
    // just
    // shadow the variable.
    public static void createNewDataOnStack(final SimulatedStack<Object> stack, final String id,
            final IndirectionDate date) {
        Object value = null;
        try {
            value = stack.currentStackFrame()
                .getValue(id);
        } catch (final ValueNotInFrameException e) {
        }

        if (value != null) {
            throw new PCMModelInterpreterException(
                    "Did expect " + id + " to not be present on stack, but found: " + value);
        }

        stack.currentStackFrame()
            .addValue(id, date);
    }

    public static SimulatedStackframe<Object> indirectionDateToStackframe(final String baseName,
            final IndirectionDate date) {
        var stackframe = new SimulatedStackframe<Object>();

        flattenDataOnStackframe(stackframe, baseName, date);

        return stackframe;
    }

    public static void flattenDataOnStack(final SimulatedStack<Object> stack, final String baseName,
            final IndirectionDate date) {
        final SimulatedStackframe<Object> currentStackframe = stack.currentStackFrame();

        flattenDataOnStackframe(currentStackframe, baseName, date);
    }

    private static void flattenDataOnStackframe(final SimulatedStackframe<Object> stackframe, final String baseName,
            final IndirectionDate date) {
        if (date instanceof ConcreteIndirectionDate) {
            final ConcreteIndirectionDate indirectionDate = (ConcreteIndirectionDate) date;

            for (final Entry<String, Object> dataEntry : indirectionDate.getData()
                .entrySet()) {
                stackframe.addValue(baseName + "." + dataEntry.getKey(), dataEntry.getValue());
            }
        } else if (date instanceof ConcreteGroupingIndirectionDate<?>) {
            final ConcreteGroupingIndirectionDate<?> groupingIndirectionDate = (ConcreteGroupingIndirectionDate<?>) date;

            for (final Entry<String, Object> dataEntry : groupingIndirectionDate.getData()
                .entrySet()) {
                stackframe.addValue(baseName + "." + dataEntry.getKey(), dataEntry.getValue());
            }
        } else if (date instanceof GenericJoinedDate) {
            final GenericJoinedDate<?, ?> genericJoinedDate = (GenericJoinedDate<?, ?>) date;

            for (var entry : genericJoinedDate.data.entrySet()) {
                flattenDataOnStackframe(stackframe, baseName + "." + entry.getKey(), entry.getValue().date);
            }
        } else {
            throw new PCMModelInterpreterException(
                    baseName + " is not a ConcreteIndirectionDate, but a " + date.getClass()
                        .getName());
        }
    }

    public static Parameter getOneParameter(final EventGroup eventGroup) {
        return IterableUtil.claimOne(eventGroup.getEventTypes__EventGroup())
            .getParameter__EventType();
    }

    public static <T extends Entity> T initName(final T entity, final String name) {
        entity.setEntityName(name);
        entity.setId(name + ".ID");

        return entity;
    }

    public static <T extends Identifier> T initName(final T identifier, final String name) {
        identifier.setId(name + ".ID");
        return identifier;
    }

    public static void makeDateInformationAvailableOnStack(final SimulatedStack<Object> stack,
            final String referenceName) {
        final SimulatedStackframe<Object> currentStackframe = stack.currentStackFrame();

        Object o;
        try {
            o = currentStackframe.getValue(referenceName);
        } catch (final ValueNotInFrameException e) {
            e.printStackTrace();
            throw new PCMModelInterpreterException(referenceName + " not found on stack.", e);
        }

        flattenDataOnStack(stack, referenceName, (IndirectionDate) o);
    }

    /**
     * Changes the prefix in the given variable name from the parameter name of the incoming event
     * type to the one of the outgoing event type.
     */
    public static String rewriteVariableNamePrefix(final String variableName, final String incomingParameterName,
            final String outgoingParameterName) {
        if (variableName.startsWith(incomingParameterName)) {
            return outgoingParameterName + variableName.substring(incomingParameterName.length());
        } else {
            throw new AssertionError("Variable '" + variableName + "' does not start with incoming paramete name: "
                    + incomingParameterName);
        }
    }

    public static PeriodicallyTriggeredSimulationEntity triggerPeriodically(final SimuComModel model,
            final double firstOccurrence, final double delay, final Runnable taskToRun) {

        return new PeriodicallyTriggeredSimulationEntity(model.getSimEngineFactory(), firstOccurrence, delay) {
            @Override
            protected void triggerInternal() {
                System.out.println("Triggering periodic process at " + model.getSimulationControl()
                    .getCurrentSimulationTime());
                taskToRun.run();
            }
        };
    }

    public static SimuComEntity triggerOnce(SimuComModel model, double delay, Runnable taskToRun) {
        return new OneShotSimulationEntity(model, delay) {
            @Override
            protected void triggerInternal() {
                System.out.println("Triggering single process at " + model.getSimulationControl()
                    .getCurrentSimulationTime());
                taskToRun.run();
            }
        };
    }

    public static void validateIndirectionDateStructure(final IndirectionDate date, final DataInterface dataInterface) {
        // TODO implement validation
        LOGGER.debug("Not validating indirection date structure");
    }

    public static void validateStackframeStructure(final Map<String, Object> dataMap, final String parameterName) {
        final String parameterCharacterisationPrefix = parameterName + ".";

        for (final Entry<String, Object> entry : dataMap.entrySet()) {
            if (!entry.getKey()
                .startsWith(parameterCharacterisationPrefix)) {
                throw new IllegalArgumentException("Invalid characteristation for data frame: " + entry.getKey()
                        + ", expected characteristations for " + parameterName);
            }
        }
    }

    private IndirectionSimulationUtil() {
    }

    public static void requireNumberOfSinkSourceRoles(DataChannel dataChannel, Predicate<Integer> sinkRoleCheck,
            String sinkRoleCheckDescription, Predicate<Integer> sourceRoleCheck, String sourceRoleCheckDescription) {

        int numberOfSinkRoles = dataChannel.getDataSinkRoles()
            .size();
        int numberOfSourceRoles = dataChannel.getDataSourceRoles()
            .size();

        if (!sinkRoleCheck.test(numberOfSinkRoles)) {
            throw new PCMModelInterpreterException("Number of sink roles not suitable for " + dataChannel
                    + ": expected " + sinkRoleCheckDescription + ", got " + numberOfSinkRoles);
        }
        if (!sourceRoleCheck.test(numberOfSourceRoles)) {
            throw new PCMModelInterpreterException("Number of source roles not suitable for " + dataChannel
                    + ": expected " + sourceRoleCheckDescription + ", got " + numberOfSourceRoles);
        }
    }

    public static void requireExactNumberOfSinkSourceRoles(DataChannel dataChannel, int sinkRoleNumber,
            int sourceRoleNumber) {
        requireNumberOfSinkSourceRoles(dataChannel, it -> it == sinkRoleNumber, "== " + sinkRoleNumber,
                it -> it == sourceRoleNumber, "== " + sourceRoleNumber);
    }

    public static double getDoubleParameter(JavaClassDataChannel dataChannel, String parameterName) {
        return Double.valueOf(forceGetParameter(parameterName, dataChannel));
    }

    public static int getIntegerParameter(JavaClassDataChannel dataChannel, String parameterName) {
        return Integer.valueOf(forceGetParameter(parameterName, dataChannel));
    }

    public static boolean getBooleanParameter(JavaClassDataChannel dataChannel, String parameterName) {
        return Boolean.valueOf(forceGetParameter(parameterName, dataChannel));
    }

    public static String getStringParameter(JavaClassDataChannel dataChannel, String parameterName) {
        return forceGetParameter(parameterName, dataChannel);
    }

    public static String forceGetParameter(String parameterName, JavaClassDataChannel dataChannel) {
        var configEntries = dataChannel.getConfigEntries();
        return Objects.requireNonNull(toConfigMap(configEntries).get(parameterName), "Could not find parameter "
                + parameterName + " in configuration (" + String.join(", ", configEntries) + ")");
    }

    public static Map<String, String> toConfigMap(Collection<String> entries) {
        return entries.stream()
            .map(it -> forceSplitAtFirstOccurence(it, "->"))
            .collect(Collectors.toMap(it -> it[0], it -> it[1]));
    }

    private static String[] forceSplitAtFirstOccurence(String it, String splitter) {
        var split = it.indexOf(splitter);
        if (split == -1)
            throw new PCMModelInterpreterException("Cannot split string " + it + " at " + splitter);

        return new String[] { it.substring(0, split), it.substring(split + splitter.length()) };
    }

    @SuppressWarnings("unchecked")
    public static void flatResolveTimes(Object timeDependency, List<Double> times) {
        if (timeDependency instanceof Collection) {
            ((Collection<Object>) timeDependency).forEach(it -> flatResolveTimes(it, times));
        } else if (timeDependency instanceof ConcreteIndirectionDate) {
            times.addAll(((ConcreteIndirectionDate) timeDependency).getTime());
        } else {
            throw new PCMModelInterpreterException("Error when getting date for " + timeDependency);
        }
    }
    
    public static List<Double> flatResolveTimes(List<Object> timeDependencies) {
        List<Double> result = new ArrayList<Double>();
        for (var it : timeDependencies) {
            flatResolveTimes(it, result);
        }
        return result;
    }

}
