package org.palladiosimulator.textual.commons.generator.rules.impl;

import com.google.common.base.Objects;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.palladiosimulator.textual.commons.generator.registry.ChildPropertyMapping;
import org.palladiosimulator.textual.commons.generator.registry.GeneratorTransformationRegistry;
import org.palladiosimulator.textual.commons.generator.rules.ExecutableRule;

/**
 * A registration denotes a registered transformation of an object of type A to an object of type B.
 * This registration also contains all the necessary instructions for when a transformation should happen
 * (what is called a "predicate" here), how the target is created as well as all what transformation steps need to happen.
 */
@SuppressWarnings("all")
public class Registration<S extends Object, T extends Object> implements ExecutableRule<S, T> {
  private static final Predicate<Object> DEFAULT_PREDICATE = ((Predicate<Object>) (Object it) -> {
    return true;
  });

  private final Class<S> mappingSource;

  private final Class<T> mappingTarget;

  private Function<S, T> factory;

  private Predicate<S> predicate = ((Predicate<S>) Registration.DEFAULT_PREDICATE);

  private Consumer<T> callback = ((Consumer<T>) (T it) -> {
  });

  private final List<ChildPropertyMapping<S, T>> contents = new ArrayList<ChildPropertyMapping<S, T>>();

  public Registration(final Class<S> source, final Class<T> target, final Function<S, T> factory) {
    this(source, target);
    this.factory = factory;
  }

  public Registration(final Class<S> source, final Class<T> target) {
    this.mappingSource = source;
    this.mappingTarget = target;
  }

  public Class<S> getSourceType() {
    return this.mappingSource;
  }

  public Class<T> getTargetType() {
    return this.mappingTarget;
  }

  /**
   * Tests whether this registration can be used to transform the given source object.
   * 
   * @param instance The object instance to check the mapping for
   */
  public boolean doesMap(final Object instance) {
    return (this.mappingSource.isAssignableFrom(instance.getClass()) && this.predicate.test(this.mappingSource.cast(instance)));
  }

  /**
   * Tests whether this registration can be used to transform the given source object into the given
   * target type. If the target type is null, this behaves the same as doesMap(Object).
   * 
   * @param instance The object instance to check the transformation for
   * @param target The desired resulting type of the transformation
   * 
   * @see #doesMap(Object)
   */
  @Override
  public boolean appliesTo(final Object instance, final Class<?> target) {
    return (((target == null) || (this.mappingTarget == target)) && this.doesMap(instance));
  }

  /**
   * Checks whether the registration has a predicate.
   */
  public boolean hasPredicate() {
    return (this.predicate != Registration.DEFAULT_PREDICATE);
  }

  public boolean hasFactory() {
    return (this.factory != null);
  }

  /**
   * Creates the target object from the source object using the configured factory.
   * 
   * @see #setFactory(Function)
   */
  public T create(final S source) {
    return this.factory.apply(source);
  }

  /**
   * Adds a transformation for a child property of the source object.
   */
  public boolean addPropertyMapping(final ChildPropertyMapping<S, T> mapping) {
    return this.contents.add(mapping);
  }

  public Predicate<S> setPredicate(final Predicate<S> pred) {
    return this.predicate = pred;
  }

  public Function<S, T> setFactory(final Function<S, T> factory) {
    return this.factory = factory;
  }

  public Consumer<T> setCallback(final Consumer<T> callback) {
    return this.callback = callback;
  }

  /**
   * Applies the complete transformation of the source to the configured target type.
   * 
   * @param source The source object to transform.
   * @param registry The registry to use for resolving possible transformations of child elements.
   * @return The transformed target object.
   */
  @Override
  public void applyTo(final S source, final T target, final GeneratorTransformationRegistry registry) {
    try {
      final Consumer<ChildPropertyMapping<S, T>> _function = (ChildPropertyMapping<S, T> it) -> {
        it.run(source, target, registry);
      };
      this.contents.forEach(_function);
    } catch (final Throwable _t) {
      if (_t instanceof Exception) {
        final Exception e = (Exception)_t;
        String _simpleName = source.getClass().getSimpleName();
        String _plus = ("Exception occurred when transforming object of type " + _simpleName);
        String _plus_1 = (_plus + " to ");
        String _simpleName_1 = target.getClass().getSimpleName();
        String _plus_2 = (_plus_1 + _simpleName_1);
        throw new RuntimeException(_plus_2, e);
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    this.callback.accept(target);
  }

  @Override
  public <U extends Object, V extends Object> boolean conflictsWith(final ExecutableRule<U, V> conflictCandidate) {
    boolean _xblockexpression = false;
    {
      if ((conflictCandidate instanceof Registration)) {
        if ((((Objects.equal(((Registration)conflictCandidate).getSourceType(), this.getSourceType()) && Objects.equal(((Registration)conflictCandidate).getTargetType(), this.getTargetType())) && (!this.hasPredicate())) && (!((Registration)conflictCandidate).hasPredicate()))) {
          return true;
        }
      }
      _xblockexpression = false;
    }
    return _xblockexpression;
  }
}
