package org.palladiosimulator.analyzer.workflow.core.blackboard;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.palladiosimulator.pcm.allocation.Allocation;
import org.palladiosimulator.pcm.allocation.AllocationPackage;
import org.palladiosimulator.pcm.repository.Repository;
import org.palladiosimulator.pcm.repository.RepositoryPackage;
import org.palladiosimulator.pcm.resourceenvironment.ResourceEnvironment;
import org.palladiosimulator.pcm.resourceenvironment.ResourceenvironmentPackage;
import org.palladiosimulator.pcm.resourcetype.ResourceRepository;
import org.palladiosimulator.pcm.resourcetype.ResourcetypePackage;
import org.palladiosimulator.pcm.system.System;
import org.palladiosimulator.pcm.system.SystemPackage;
import org.palladiosimulator.pcm.usagemodel.UsageModel;
import org.palladiosimulator.pcm.usagemodel.UsagemodelPackage;

import de.uka.ipd.sdq.featureconfig.Configuration;
import de.uka.ipd.sdq.featureconfig.featureconfigPackage;
import de.uka.ipd.sdq.workflow.blackboard.Blackboard;
import de.uka.ipd.sdq.workflow.mdsd.blackboard.ResourceSetPartition;

/**
 * This class is a specialised MDSDBlackboard partition which is specialised to load and hold PCM
 * model instances. Currently, the PCM instance is loaded as a list of files, each file containing a
 * part of the PCM model. It is sufficient to specify a PCM Allocation and a PCM UsageModel. All
 * other model parts can then be derived automatically.
 *
 * Note that there is no specialised {@link Blackboard} for PCMResourceSetPartitions, because
 * usually, the Blackboard containing the PCM might contain other models, for example SAAM models
 * when transforming one into the other.
 *
 *
 * @author Steffen Becker
 *
 */
public class PCMResourceSetPartition extends ResourceSetPartition {

    /**
     * @return Returns the PCM repository instance of the PCM model
     */
    public List<Repository> getRepositories() {
        // TODO: Allow using multiple Repositories, derive the list of repositories automatically
        final List<Repository> allRepositories = getElement(RepositoryPackage.eINSTANCE.getRepository());
        final Iterator<Repository> iterator = allRepositories.iterator();
        final List<Repository> resultList = new ArrayList<Repository>();
        while (iterator.hasNext()) {
            resultList.add(iterator.next());
        }
        return resultList;
    }

    /**
     * @return Returns a PCM Repository which contains components of Steffen's and Jens' middleware
     *         completions
     */
    public Repository getMiddlewareRepository() {
        return (Repository) getElement(RepositoryPackage.eINSTANCE.getRepository()).get(0);
    }

    /**
     * @return Returns the feature configuration which annotates connectors with their technical
     *         realisation
     */
    public Configuration getFeatureConfig() {
        return (Configuration) getElement(featureconfigPackage.eINSTANCE.getConfiguration()).get(0);
    }

    /**
     * @return Returns the PCM system instance of the stored PCM model
     */
    public System getSystem() {
        return (System) getElement(SystemPackage.eINSTANCE.getSystem()).get(0);
    }

    /**
     * @return Returns the PCM system's allocation model
     */
    public Allocation getAllocation() {
        return (Allocation) getElement(AllocationPackage.eINSTANCE.getAllocation()).get(0);
    }

    /**
     * @return Returns the PCM usage model of the PCM model in this blackboard partition
     */
    public UsageModel getUsageModel() {
        return (UsageModel) getElement(UsagemodelPackage.eINSTANCE.getUsageModel()).get(0);
    }

    /**
     * @return Returns the PCM Resource Type Repository used by the stored PCM model instance
     */
    public ResourceRepository getResourceTypeRepository() {
        return (ResourceRepository) getElement(ResourcetypePackage.eINSTANCE.getResourceRepository()).get(0);
    }

    /**
     * @return Returns the PCM Resource Environment used by the stored PCM model instance
     */
    public ResourceEnvironment getResourceEnvironment() {
        return (ResourceEnvironment) getElement(ResourceenvironmentPackage.eINSTANCE.getResourceEnvironment()).get(0);
    }

}
