/**
 *
 */
package org.palladiosimulator.analyzer.accuracy.transformation;

import static org.palladiosimulator.analyzer.quality.util.StoExHelper.*;

import org.apache.log4j.Logger;

import org.palladiosimulator.analyzer.quality.qualityannotation.ExactlyAsSpecifiedPrecision;
import org.palladiosimulator.analyzer.quality.qualityannotation.LimitedDeviationPrecision;
import org.palladiosimulator.analyzer.quality.qualityannotation.NoPrecision;
import org.palladiosimulator.analyzer.quality.qualityannotation.Precision;
import org.palladiosimulator.analyzer.quality.qualityannotation.util.QualityAnnotationSwitch;

import org.palladiosimulator.pcm.core.PCMRandomVariable;

/**Modifies a given PCM Random Variable according to the minimum allowed by a precision.
 * First set the random variable by {@link #setModifiedVariable(PCMRandomVariable)} then transform it by {@link PCMRandomVariableSpecificationAccuracyMinimumTrafo#doSwitch(org.eclipse.emf.ecore.EObject)}.

 * @author groenda
 *
 */
public class PCMRandomVariableSpecificationAccuracyMinimumTrafo extends
		QualityAnnotationSwitch<Boolean> {
	/** Logger for this class. */
	private static final Logger LOGGER = Logger.getLogger(PCMRandomVariableSpecificationAccuracyMinimumTrafo.class);

	/** The variable modified by the transformation when {@link #doSwitch(org.eclipse.emf.ecore.EObject)} is called. */
	private PCMRandomVariable modifiedVariable;

	/** Minimum value reported will not be smaller than this bound.*/
	private String lowerLimit;

	/**Sets the lower bound for the minimum.
	 * The transformation will not result in values below this bound.
	 * The bound is automatically reset after an invocation of {@link #doSwitch(org.eclipse.emf.ecore.EObject)}.
	 * @param lowerLimit The lower bound.
	 */
	public void setLowerLimit(String lowerLimit) {
		this.lowerLimit = lowerLimit;
	}

	/**Set the variable to modify upon invocation of {@link #doSwitch(org.eclipse.emf.ecore.EObject)}.
	 * @param modifiedVariable The variable.
	 */
	public void setModifiedVariable(PCMRandomVariable modifiedVariable) {
		this.modifiedVariable = modifiedVariable;
	}

	@Override
	public Boolean caseNoPrecision(NoPrecision object) {
		checkModifiedVariable();
		if (lowerLimit == null) {
			String msg = "Dynamic type inference of specifications to determine minimal value is not implemented (yet).";
			LOGGER.error(msg);
			throw new IllegalArgumentException(msg);
		} else {
			modifiedVariable.setSpecification(lowerLimit);
		}
		reset();
		return true;
	}

	@Override
	public Boolean caseExactlyAsSpecifiedPrecision(
			ExactlyAsSpecifiedPrecision object) {
		checkModifiedVariable();
		reset();
		// nothing to do
		return true;
	}

	@Override
	public Boolean caseLimitedDeviationPrecision(
			LimitedDeviationPrecision object) {
		checkModifiedVariable();
		// newSpec = max(min(noc.specification - absolute, noc.specification - noc.specification * relative, 0);
		String spec = modifiedVariable.getSpecification();
		String abs = "" + object.getAbsolute();
		String rel = "" + object.getRelative();
		String min = "MinDeviation(" + spec + ", " + abs + ", " + rel + ")";
		if (lowerLimit == null) {
			modifiedVariable.setSpecification(min);
		} else {
			modifiedVariable.setSpecification(stoExMax(min, lowerLimit));
		}
		reset();
		return true;
	}

	@Override
	public Boolean casePrecision(Precision object) {
		checkModifiedVariable();
		String msg = "The handling of the provided Precision " + object + " is not implemented.";
		LOGGER.error(msg);
		reset();
		throw new IllegalArgumentException(msg);
	}

	/**Checks that the modified variable is set.
	 *
	 */
	private void checkModifiedVariable() {
		if (modifiedVariable == null) {
			String msg = "You have to set the modified variable before invoking doSwitch(). Note: The modified variable is reset after an execution of doSwitch() to prevent accidental transformations of the same variable.";
			LOGGER.error(msg);
			throw new IllegalArgumentException(msg);
		}
	}

	/**Prevent accidental re-transformation by resetting this classes instance variables.
	 */
	private void reset() {
		modifiedVariable = null;
		lowerLimit = null;
	}
}
