package de.uka.ipd.sdq.simucomframework.usage;

import java.util.List;

import org.apache.log4j.Logger;
import org.palladiosimulator.probeframework.measurement.RequestContext;
import org.palladiosimulator.probeframework.probes.Probe;
import org.palladiosimulator.probeframework.probes.TriggeredProbe;

import de.uka.ipd.sdq.scheduler.resources.active.IResourceTableManager;
import de.uka.ipd.sdq.simucomframework.core.Context;
import de.uka.ipd.sdq.simucomframework.core.SimuComSimProcess;
import de.uka.ipd.sdq.simucomframework.core.exceptions.FailureException;
import de.uka.ipd.sdq.simucomframework.core.model.SimuComModel;
import de.uka.ipd.sdq.simucomframework.core.usage.IScenarioRunner;
import de.uka.ipd.sdq.simucomframework.core.usage.IUser;

/**
 * A closed workload user is a user which performs the typical closed workload cycle: execute,
 * think, execute, ...
 *
 * @author Steffen Becker, Sebastian Lehrig
 */
public class ClosedWorkloadUser extends SimuComSimProcess implements IUser {

    private static final Logger LOGGER = Logger.getLogger(ClosedWorkloadUser.class.getName());

    private final IScenarioRunner scenarioRunner;
    private String thinkTime;

    // private static int USERCOUNT = 0;
    private int runCount = 0;

    private final List<Probe> usageStartStopProbes;

    /**
     * Constructor of the closed workload user
     *
     * @param owner
     *            The model this user belongs to
     * @param name
     *            The users name
     * @param scenarioRunner
     *            The scenario runner determining the users behaviour
     * @param thinkTimeSpec
     *            A stoex which determines the users think time
     * @param usageStartStopProbes
     */
    public ClosedWorkloadUser(final SimuComModel owner, final String name, final IScenarioRunner scenarioRunner,
            final String thinkTimeSpec, final List<Probe> usageStartStopProbes, IResourceTableManager resourceTableManager) {
        super(owner, name, resourceTableManager);
        this.scenarioRunner = scenarioRunner;
        this.thinkTime = thinkTimeSpec;
        this.usageStartStopProbes = usageStartStopProbes;
    }

    protected long sessionId;

    private boolean requestStop = false;

    /*
     * (non-Javadoc)
     * 
     * @see desmoj.core.simulator.SimProcess#lifeCycle()
     */
    @Override
    protected void internalLifeCycle() {

        // Repeat usage scenario runs as long as simulation is running:
        while (!requestStop && getModel().getSimulationControl().isRunning()) {
            // update session id
            updateNewSessionID();

            try {
                if (getModel().getConfiguration().getSimulateFailures()) {
                    this.getModel().getFailureStatistics().increaseRunCount();
                    if (getModel().getConfiguration().isDebug()) {
                        this.getModel().getFailureStatistics().printRunCount(LOGGER,
                                getModel().getSimulationControl().getCurrentSimulationTime());
                    }
                }
                scenarioRunner(this);
                if (getModel().getConfiguration().getSimulateFailures()) {
                    this.getModel().getFailureStatistics().recordSuccess();
                }
            } catch (final FailureException exception) {
                if (getModel().getConfiguration().getSimulateFailures()) {
                    this.getModel().getFailureStatistics().increaseUnhandledFailureCounter(exception.getFailureType(),
                            currentSessionId);
                }
            } finally {
                // Increase measurements counter manually as usage scenario run
                // is not finished:
                this.getModel().increaseMainMeasurementsCount();

                runCount++;
            }
        }

        // Mark user as finished:
        // USERCOUNT--;

        // Print failure statistics after last user is finished:
        // if (USERCOUNT == 0) {
        // if (this.getModel().getConfig().getSimulateFailures()) {
        // FailureStatistics.getInstance().printFailureStatistics(LOGGER);
        // }
        // }
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.uka.ipd.sdq.simucomframework.usage.IScenarioRunner#scenarioRunner(
     * desmoj.core.simulator.SimProcess)
     */
    @Override
    public void scenarioRunner(final SimuComSimProcess thread) {
        // execute
        ((TriggeredProbe) this.usageStartStopProbes.get(0)).takeMeasurement(getRequestContext());
        this.scenarioRunner.scenarioRunner(thread);
        ((TriggeredProbe) this.usageStartStopProbes.get(1)).takeMeasurement(getRequestContext());
        // think
        final double thinkTimeEvaluation = Context.evaluateStatic(this.thinkTime, Double.class, null);
        this.hold(thinkTimeEvaluation);
    }

    /*
     * (non-Javadoc)
     * 
     * @see de.uka.ipd.sdq.simucomframework.usage.IUser#startUserLife()
     */
    @Override
    public void startUserLife() {
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug(this.getName() + " started! I'm alive!!!");
        }
        // USERCOUNT++;
        this.scheduleAt(0);
    }

    /**
     * {@inheritDoc}
     * <p>
     * A closed workload user runs through his lifecycle several times in series. Each time the same
     * {@link SimuComSimProcess} is used. In order to be able to distinguish between the different
     * runs, the request context id gets concatenated with the run count. The run counts represents
     * how often the lifecycle has been executed. Thus the returned ID is unique for each run
     * through the lifecycle.
     */
    @Override
    public RequestContext getRequestContext() {
        return super.getRequestContext().append("." + runCount);
    }

    public void requestStop() {
        this.requestStop = true;
    }

    public void setThinkTime(final String newThinkTimeSpec) {
        this.thinkTime = newThinkTimeSpec;
    }
}
