package org.palladiosimulator.pcm.confidentiality.attacker.helper;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.AttackerSystemSpecificationContainer;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.Attack;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.AttackVector;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.ConfidentialityImpact;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.Role;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.Vulnerability;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.pcmIntegration.PCMElement;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.pcmIntegration.RoleSystemIntegration;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.pcmIntegration.VulnerabilitySystemIntegration;
import org.palladiosimulator.pcm.confidentiality.context.system.pcm.structure.MethodSpecification;
import org.palladiosimulator.pcm.core.composition.AssemblyContext;
import org.palladiosimulator.pcm.resourceenvironment.LinkingResource;
import org.palladiosimulator.pcm.resourceenvironment.ResourceContainer;

public final class VulnerabilityHelper {
    private VulnerabilityHelper() {
        assert false;
    }

    private static HashMap<String, List<Vulnerability>> vulnarabilityMap = new HashMap<>();

    /**
     * Checks whether an attack is possible and returns the {@link Vulnerability} with the highest
     * impact on Confidentiality. If the vulnerabilities contain multiple vulnerabilities with
     * highest impact it returns any vulnerability of it
     *
     * @param credentials
     * @param policies
     * @param vulnerabilities
     * @param attacks
     * @param vector
     * @return {@link Vulnerability} with the highest ConfidentialityImpact if no attack is possible
     *         null
     */
    public static Vulnerability checkAttack(final boolean authenticated, final List<Vulnerability> vulnerabilities,
            final List<Attack> attacks, final AttackVector vector, final List<Role> roles) {
        final var applicableVulnerabilities = new ArrayList<Vulnerability>();
        for (final var vulnerability : vulnerabilities) {
            for (final var attack : attacks) {

                if (attack.canExploit(vulnerability, authenticated, vector)) {
                    if (vulnerability.getConfidentialityImpact() == ConfidentialityImpact.HIGH) {
                        return vulnerability;
                    }
                    applicableVulnerabilities.add(vulnerability);
                }

            }
        }
        final var returnValue = applicableVulnerabilities.stream()
            .filter(e -> e.getConfidentialityImpact() == ConfidentialityImpact.LOW)
            .findAny();
        if (returnValue.isPresent()) {
            return returnValue.get();
        }
        return applicableVulnerabilities.stream()
            .findAny()
            .orElse(null);
    }

    public static void initializeVulnerabilityStorage(
            final AttackerSystemSpecificationContainer vulnerabilityspecification) {
        resetMap();

        final var listVulnerabilitySystemIntegration = filterSystemSpecification(vulnerabilityspecification);

        listVulnerabilitySystemIntegration.parallelStream()
            .filter(e -> e.getPcmelement() != null)
            .forEach(e -> {
                if (e.getPcmelement()
                    .getResourcecontainer() != null) {
                    putInMap(e.getPcmelement()
                        .getResourcecontainer()
                        .getId(),
                            getVulnerabilities(listVulnerabilitySystemIntegration, e.getPcmelement()
                                .getResourcecontainer()));
                } else if (e.getPcmelement()
                    .getLinkingresource() != null) {
                    putInMap(e.getPcmelement()
                        .getLinkingresource()
                        .getId(),
                            getVulnerabilities(listVulnerabilitySystemIntegration, e.getPcmelement()
                                .getLinkingresource()));
                } else if (e.getPcmelement()
                    .getAssemblycontext() != null
                        && !e.getPcmelement()
                            .getAssemblycontext()
                            .isEmpty()) {
                    putInMap(e.getPcmelement()
                        .getAssemblycontext()
                        .get(0)
                        .getId(),
                            getVulnerabilities(listVulnerabilitySystemIntegration, e.getPcmelement()
                                .getAssemblycontext()
                                .get(0)));
                } else if (e.getPcmelement()
                    .getMethodspecification() != null) {
                    putInMap(e.getPcmelement()
                        .getMethodspecification()
                        .getId(),
                            getVulnerabilities(listVulnerabilitySystemIntegration, e.getPcmelement()
                                .getMethodspecification()));
                }

            });

//        listVulnerabilitySystemIntegration.parallelStream()
//				.filter(e -> e.getPcmelement() != null && e.getPcmelement().getLinkingresource() != null)
//				.forEach(e -> putInMap(e.getPcmelement().getLinkingresource().getId(), getVulnerabilities(
//						listVulnerabilitySystemIntegration, e.getPcmelement().getLinkingresource())));
//
//        listVulnerabilitySystemIntegration.parallelStream()
//				.filter(e -> e.getPcmelement() != null && e.getPcmelement().getAssemblycontext() != null
//						&& !e.getPcmelement().getAssemblycontext().isEmpty())
//				.forEach(e -> putInMap(e.getPcmelement().getAssemblycontext().get(0).getId(), getVulnerabilities(
//						listVulnerabilitySystemIntegration, e.getPcmelement().getAssemblycontext().get(0))));
//
//        listVulnerabilitySystemIntegration.parallelStream()
//				.filter(e -> e.getPcmelement() != null && e.getPcmelement().getMethodspecification() != null)
//				.forEach(e -> putInMap(e.getPcmelement().getMethodspecification().getId(), getVulnerabilities(
//						listVulnerabilitySystemIntegration, e.getPcmelement().getMethodspecification())));
    }

    public static void resetMap() {
        vulnarabilityMap = new HashMap<>();
        vulnarabilityMap.clear();
    }

    // IMPORTANT: HashMap is unsynchronized, so synchronization must be done here.
    private static synchronized void putInMap(final String key, final List<Vulnerability> value) {
        vulnarabilityMap.put(key, value);
    }

    public static List<Vulnerability> getVulnerabilities(
            final AttackerSystemSpecificationContainer vulnerabilityspecification, final ResourceContainer resource) {
        if (!vulnarabilityMap.containsKey(resource.getId())) {
            return new ArrayList<>();
        }

        return vulnarabilityMap.get(resource.getId());
    }

    private static List<VulnerabilitySystemIntegration> filterSystemSpecification(
            final AttackerSystemSpecificationContainer vulnerabilityspecification) {
        return vulnerabilityspecification.getVulnerabilities()
            .stream()
            .filter(VulnerabilitySystemIntegration.class::isInstance)
            .map(VulnerabilitySystemIntegration.class::cast)
            .collect(Collectors.toList());
    }

    public static List<RoleSystemIntegration> getRoles(
            final AttackerSystemSpecificationContainer vulnerabilityspecification) {
        return vulnerabilityspecification.getVulnerabilities()
            .stream()
            .filter(RoleSystemIntegration.class::isInstance)
            .map(RoleSystemIntegration.class::cast)
            .collect(Collectors.toList());
    }

    public static List<Vulnerability> getVulnerabilities(
            final List<VulnerabilitySystemIntegration> vulnerabilitySpecification, final ResourceContainer resource) {
        return getVulnerbilities(vulnerabilitySpecification, PCMElement::getResourcecontainer, resource);
    }

    public static List<Vulnerability> getVulnerabilities(
            final AttackerSystemSpecificationContainer vulnerabilityspecification, final AssemblyContext component) {
        if (!vulnarabilityMap.containsKey(component.getId())) {
            return new ArrayList<>();
        }

        return vulnarabilityMap.get(component.getId());
    }

    public static List<Vulnerability> getVulnerabilities(
            final AttackerSystemSpecificationContainer vulnerabilityspecification,
            final MethodSpecification methodSpecification) {
        if (!vulnarabilityMap.containsKey(methodSpecification.getId())) {
            return new ArrayList<>();
        }

        return vulnarabilityMap.get(methodSpecification.getId());
    }

    public static List<Vulnerability> getVulnerabilities(
            final List<VulnerabilitySystemIntegration> vulnerabilitySpecification,
            final MethodSpecification methodSpecification) {
        return getVulnerbilities(vulnerabilitySpecification, PCMElement::getMethodspecification, methodSpecification);
    }

    public static List<Vulnerability> getVulnerabilities(
            final List<VulnerabilitySystemIntegration> vulnerabilitySpecification, final AssemblyContext component) {
        return vulnerabilitySpecification.stream()
            .filter(target -> filterAssemblyContexts(target.getPcmelement()
                .getAssemblycontext(), List.of(component)))
            .map(VulnerabilitySystemIntegration::getVulnerability)
            .collect(Collectors.toList());
    }

    public static List<Vulnerability> getVulnerabilities(
            final AttackerSystemSpecificationContainer vulnerabilityspecification, final LinkingResource resource) {
        if (!vulnarabilityMap.containsKey(resource.getId())) {
            return new ArrayList<>();
        }

        return vulnarabilityMap.get(resource.getId());
    }

    public static List<Vulnerability> getVulnerabilities(
            final List<VulnerabilitySystemIntegration> vulnerabilitySpecification, final LinkingResource resource) {
        return getVulnerbilities(vulnerabilitySpecification, PCMElement::getLinkingresource, resource);
    }

    private static List<Vulnerability> getVulnerbilities(final List<VulnerabilitySystemIntegration> specification,
            final Function<PCMElement, EObject> method, final EObject object) {
        return specification.stream()
            .filter(e -> EcoreUtil.equals(method.apply(e.getPcmelement()), object))
            .map(VulnerabilitySystemIntegration::getVulnerability)
            .collect(Collectors.toList());
    }

    private static boolean filterAssemblyContexts(final List<AssemblyContext> target,
            final List<AssemblyContext> component) {
        if (target.size() != component.size()) {
            return false;
        }
        for (final AssemblyContext element : target) {
            if (!EcoreUtil.equals(element, component.get(0))) {
                return false;
            }
        }
        return true;
    }

}
