package org.palladiosimulator.retriever.extraction.rules;

import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;
import org.apache.log4j.Logger;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.IMethodBinding;
import org.eclipse.jdt.core.dom.ITypeBinding;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.SingleVariableDeclaration;
import org.eclipse.jdt.core.dom.Type;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.jdom2.Document;
import org.palladiosimulator.retriever.extraction.commonalities.CompUnitOrName;
import org.palladiosimulator.retriever.extraction.commonalities.HTTPMethod;
import org.palladiosimulator.retriever.extraction.commonalities.RESTName;
import org.palladiosimulator.retriever.extraction.commonalities.RESTOperationName;
import org.palladiosimulator.retriever.extraction.engine.PCMDetector;
import org.palladiosimulator.retriever.extraction.engine.RuleHelper;
import org.palladiosimulator.retriever.extraction.rules.util.ProjectHelper;
import org.palladiosimulator.retriever.extraction.rules.util.RESTHelper;
import org.palladiosimulator.retriever.extraction.rules.util.SpringHelper;
import org.palladiosimulator.retriever.services.Rule;
import org.palladiosimulator.retriever.services.blackboard.RetrieverBlackboard;

@SuppressWarnings("all")
public class SpringRules implements Rule {
  private static final Logger LOG = Logger.getLogger(SpringRules.class);

  public static final String RULE_ID = "org.palladiosimulator.retriever.extraction.rules.spring";

  public static final String JAVA_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.java";

  public static final String YAML_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.yaml";

  public static final String YAML_MAPPERS_KEY = (SpringRules.YAML_DISCOVERER_ID + ".mappers");

  public static final String XML_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.xml";

  public static final String PROPERTIES_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.properties";

  public static final String ZUUL_RULE_ID = "org.palladiosimulator.retriever.extraction.rules.spring.zuul";

  public static final String CLOUD_GATEWAY_RULE_ID = "org.palladiosimulator.retriever.extraction.rules.spring.cloudgateway";

  @Override
  public void processRules(final RetrieverBlackboard blackboard, final Path path) {
    final CompilationUnit unit = blackboard.<CompilationUnit>getDiscoveredFiles(SpringRules.JAVA_DISCOVERER_ID, CompilationUnit.class).get(path);
    if ((unit == null)) {
      return;
    }
    Object _partition = blackboard.getPartition(SpringRules.YAML_DISCOVERER_ID);
    final Map<Path, Iterable<Map<String, Object>>> rawYamls = ((Map<Path, Iterable<Map<String, Object>>>) _partition);
    Object _partition_1 = blackboard.getPartition(SpringRules.YAML_MAPPERS_KEY);
    final Map<Path, Function<String, Optional<String>>> yamlMappers = ((Map<Path, Function<String, Optional<String>>>) _partition_1);
    final Map<Path, Document> poms = blackboard.<Document>getDiscoveredFiles(SpringRules.XML_DISCOVERER_ID, Document.class);
    final Map<Path, Properties> propertyFiles = blackboard.<Properties>getDiscoveredFiles(SpringRules.PROPERTIES_DISCOVERER_ID, Properties.class);
    final Path projectRoot = ProjectHelper.findProjectRoot(path, "pom.xml");
    final Path configRoot = SpringHelper.findConfigRoot(poms);
    Function<String, Optional<String>> _xifexpression = null;
    if ((projectRoot == null)) {
      _xifexpression = null;
    } else {
      _xifexpression = yamlMappers.get(
        SpringHelper.findFile(yamlMappers.keySet(), projectRoot.resolve("src/main/resources"), 
          Set.<String>of("bootstrap.yaml", "bootstrap.yml")));
    }
    final Function<String, Optional<String>> bootstrapYaml = _xifexpression;
    Properties _xifexpression_1 = null;
    if ((projectRoot == null)) {
      _xifexpression_1 = null;
    } else {
      _xifexpression_1 = propertyFiles.get(
        SpringHelper.findFile(propertyFiles.keySet(), projectRoot.resolve("src/main/resources"), 
          Set.<String>of("application.properties")));
    }
    final Properties applicationProperties = _xifexpression_1;
    String applicationName = SpringHelper.getFromYamlOrProperties("spring.application.name", bootstrapYaml, applicationProperties);
    if ((applicationName == null)) {
      applicationName = "SPRING-APPLICATION";
    }
    Function<String, Optional<String>> _xifexpression_2 = null;
    if ((configRoot == null)) {
      _xifexpression_2 = null;
    } else {
      _xifexpression_2 = yamlMappers.get(
        SpringHelper.findFile(yamlMappers.keySet(), configRoot.resolve("src/main/resources/shared"), 
          Set.<String>of((applicationName + ".yaml"), (applicationName + ".yml"))));
    }
    final Function<String, Optional<String>> projectConfigYaml = _xifexpression_2;
    final Function<Function<String, Optional<String>>, Optional<String>> _function = (Function<String, Optional<String>> x) -> {
      return x.apply("server.servlet.context-path");
    };
    final Optional<String> contextPathOption = Optional.<Function<String, Optional<String>>>ofNullable(projectConfigYaml).<String>flatMap(_function);
    String contextPath = contextPathOption.orElse("/");
    Iterable<Map<String, Object>> _xifexpression_3 = null;
    if ((projectRoot == null)) {
      _xifexpression_3 = null;
    } else {
      _xifexpression_3 = rawYamls.get(
        SpringHelper.findFile(yamlMappers.keySet(), projectRoot.resolve("src/main/resources"), 
          Set.<String>of("application.yaml", "application.yml")));
    }
    final Iterable<Map<String, Object>> rawApplicationYaml = _xifexpression_3;
    final Map<String, String> contextVariables = this.collectContextVariables(rawApplicationYaml);
    this.processRuleForCompUnit(blackboard, unit, applicationName, contextPath, contextVariables);
  }

  public Map<String, String> collectContextVariables(final Iterable<Map<String, Object>> applicationYaml) {
    final HashMap<String, String> result = new HashMap<String, String>();
    if (((applicationYaml == null) || IterableExtensions.isEmpty(applicationYaml))) {
      return result;
    }
    return this.collectContextVariables(((Map<String, Object>[])Conversions.unwrapArray(applicationYaml, Map.class))[0]);
  }

  public Map<String, String> collectContextVariables(final Map<String, Object> applicationYaml) {
    final HashMap<String, String> result = new HashMap<String, String>();
    if ((applicationYaml == null)) {
      return result;
    }
    Set<Map.Entry<String, Object>> _entrySet = applicationYaml.entrySet();
    for (final Map.Entry<String, Object> entry : _entrySet) {
      Object _value = entry.getValue();
      if ((_value instanceof Map)) {
        Object _value_1 = entry.getValue();
        final Map<String, Object> mapValue = ((Map<String, Object>) _value_1);
        Set<Map.Entry<String, String>> _entrySet_1 = this.collectContextVariables(mapValue).entrySet();
        for (final Map.Entry<String, String> mapEntry : _entrySet_1) {
          String _key = entry.getKey();
          String _plus = (_key + ".");
          String _key_1 = mapEntry.getKey();
          String _plus_1 = (_plus + _key_1);
          result.put(_plus_1, mapEntry.getValue());
        }
      } else {
        Object _value_2 = entry.getValue();
        if ((_value_2 instanceof List)) {
          Object _value_3 = entry.getValue();
          final List<Map<String, Object>> extendedMapValue = ((List<Map<String, Object>>) _value_3);
          for (final Map<String, Object> extendedEntry : extendedMapValue) {
            {
              Object _get = extendedEntry.get("key");
              final String extendedKey = ((String) _get);
              Object _get_1 = extendedEntry.get("value");
              String extendedValue = ((String) _get_1);
              if (((extendedKey != null) && (extendedValue != null))) {
                boolean _startsWith = extendedValue.startsWith("${");
                if (_startsWith) {
                  final int startIndex = extendedValue.indexOf(":");
                  final int endIndex = extendedValue.indexOf("}", startIndex);
                  extendedValue = extendedValue.substring((startIndex + 1), endIndex);
                }
                String _key_2 = entry.getKey();
                String _plus_2 = (_key_2 + ".");
                String _plus_3 = (_plus_2 + extendedKey);
                result.put(_plus_3, extendedValue);
              }
            }
          }
        } else {
          Object _value_4 = entry.getValue();
          if ((_value_4 instanceof String)) {
            Object _value_5 = entry.getValue();
            String stringValue = ((String) _value_5);
            boolean _startsWith = stringValue.startsWith("${");
            if (_startsWith) {
              final int startIndex = stringValue.indexOf(":");
              final int endIndex = stringValue.indexOf("}", startIndex);
              stringValue = stringValue.substring((startIndex + 1), endIndex);
            }
            result.put(entry.getKey(), stringValue);
          }
        }
      }
    }
    return result;
  }

  public void processRuleForCompUnit(final RetrieverBlackboard blackboard, final CompilationUnit unit, final String applicationName, final String contextPath, final Map<String, String> contextVariables) {
    Object _pCMDetector = blackboard.getPCMDetector();
    final PCMDetector pcmDetector = ((PCMDetector) _pCMDetector);
    if ((pcmDetector == null)) {
      return;
    }
    if ((unit == null)) {
      return;
    }
    final boolean isService = RuleHelper.isUnitAnnotatedWithName(unit, "Service");
    final boolean isController = (RuleHelper.isUnitAnnotatedWithName(unit, "RestController") || 
      RuleHelper.isUnitAnnotatedWithName(unit, "Controller"));
    final boolean isClient = RuleHelper.isUnitAnnotatedWithName(unit, "FeignClient");
    final boolean isRepository = this.isRepository(unit);
    final boolean isComponent = ((((isService || isController) || isClient) || isRepository) || 
      RuleHelper.isUnitAnnotatedWithName(unit, "Component"));
    final CompUnitOrName identifier = new CompUnitOrName(unit);
    boolean _endsWith = identifier.toString().endsWith("Test");
    if (_endsWith) {
      return;
    }
    if (((RuleHelper.isAbstraction(unit) && (!isClient)) && (!isRepository))) {
      return;
    }
    if (isComponent) {
      pcmDetector.detectComponent(identifier);
      final Predicate<MethodDeclaration> _function = (MethodDeclaration c) -> {
        return RuleHelper.isMethodAnnotatedWithName(c, "Autowired");
      };
      final Function<MethodDeclaration, Stream<SingleVariableDeclaration>> _function_1 = (MethodDeclaration c) -> {
        return RuleHelper.getParameters(c).stream();
      };
      final Predicate<SingleVariableDeclaration> _function_2 = (SingleVariableDeclaration p) -> {
        boolean _isParameterAnnotatedWith = RuleHelper.isParameterAnnotatedWith(p, "Value");
        return (!_isParameterAnnotatedWith);
      };
      final Consumer<SingleVariableDeclaration> _function_3 = (SingleVariableDeclaration p) -> {
        pcmDetector.detectRequiredInterface(identifier, p);
      };
      RuleHelper.getConstructors(unit).stream().filter(_function).<SingleVariableDeclaration>flatMap(_function_1).filter(_function_2).forEach(_function_3);
    } else {
      return;
    }
    if ((isService || isController)) {
      List<FieldDeclaration> _fields = RuleHelper.getFields(unit);
      for (final FieldDeclaration f : _fields) {
        {
          final boolean annotated = RuleHelper.isFieldAnnotatedWithName(f, "Autowired");
          if ((annotated || this.isRepository(f.getType().resolveBinding()))) {
            pcmDetector.detectRequiredInterface(identifier, f);
          }
        }
      }
      pcmDetector.detectPartOfComposite(identifier, RuleHelper.getUnitName(unit));
    }
    if (isController) {
      final String requestedUnitMapping = RuleHelper.getUnitAnnotationStringValue(unit, "RequestMapping");
      String ifaceName = contextPath;
      if ((requestedUnitMapping != null)) {
        String _ifaceName = ifaceName;
        ifaceName = (_ifaceName + requestedUnitMapping);
      }
      List<MethodDeclaration> _methods = RuleHelper.getMethods(unit);
      for (final MethodDeclaration m : _methods) {
        {
          final boolean annotated = this.hasMapping(m);
          if (annotated) {
            String requestedMapping = this.getMapping(m);
            if ((requestedMapping != null)) {
              requestedMapping = this.substituteVariables(requestedMapping, contextVariables);
              String methodName = ((ifaceName + "/") + requestedMapping);
              methodName = RESTHelper.replaceArgumentsWithWildcards(methodName);
              final HTTPMethod httpMethod = this.getHTTPMethod(m);
              IMethodBinding _resolveBinding = m.resolveBinding();
              RESTOperationName _rESTOperationName = new RESTOperationName(applicationName, methodName, httpMethod);
              pcmDetector.detectCompositeProvidedOperation(identifier, _resolveBinding, _rESTOperationName);
            }
          }
        }
      }
    }
    if (isClient) {
      String serviceIdentifier = RuleHelper.getUnitAnnotationStringValue(unit, "FeignClient", "name");
      if ((serviceIdentifier == null)) {
        serviceIdentifier = RuleHelper.getUnitAnnotationStringValue(unit, "FeignClient");
      }
      final String requestedUnitMapping_1 = RuleHelper.getUnitAnnotationStringValue(unit, "RequestMapping");
      String ifaceName_1 = "";
      if ((requestedUnitMapping_1 != null)) {
        String _ifaceName_1 = ifaceName_1;
        ifaceName_1 = (_ifaceName_1 + requestedUnitMapping_1);
      }
      List<MethodDeclaration> _methods_1 = RuleHelper.getMethods(unit);
      for (final MethodDeclaration m_1 : _methods_1) {
        {
          final boolean annotated = this.hasMapping(m_1);
          if (annotated) {
            String requestedMapping = this.getMapping(m_1);
            if ((requestedMapping != null)) {
              requestedMapping = this.substituteVariables(requestedMapping, contextVariables);
              String methodName = ((ifaceName_1 + "/") + requestedMapping);
              methodName = RESTHelper.replaceArgumentsWithWildcards(methodName);
              RESTName _rESTName = new RESTName(serviceIdentifier, methodName);
              pcmDetector.detectCompositeRequiredInterface(identifier, _rESTName);
            }
          }
        }
      }
    }
    List<Type> _allAbstractParents = RuleHelper.getAllAbstractParents(unit);
    for (final Type parent : _allAbstractParents) {
      {
        final ITypeBinding parentBinding = parent.resolveBinding();
        if (((parentBinding != null) && (!parentBinding.getName().endsWith("Repository")))) {
          pcmDetector.detectProvidedInterfaceWeakly(identifier, parentBinding);
          List<IMethodBinding> _methods_2 = RuleHelper.getMethods(parent);
          for (final IMethodBinding m_2 : _methods_2) {
            pcmDetector.detectProvidedOperationWeakly(identifier, parentBinding, m_2);
          }
        }
      }
    }
  }

  public String substituteVariables(final String string, final Map<String, String> contextVariables) {
    String result = string;
    while (result.contains("${")) {
      {
        final int startIndex = result.indexOf("${");
        final int endIndex = result.indexOf("}", startIndex);
        final String key = result.substring((startIndex + 2), endIndex);
        final String value = contextVariables.get(key);
        if ((value != null)) {
          String _substring = result.substring(0, startIndex);
          String _plus = (_substring + value);
          String _substring_1 = result.substring((endIndex + 1));
          String _plus_1 = (_plus + _substring_1);
          result = _plus_1;
        } else {
          String _substring_2 = result.substring(0, startIndex);
          String _plus_2 = (_substring_2 + "ERROR_COULD_NOT_RESOLVE");
          String _substring_3 = result.substring((endIndex + 1));
          String _plus_3 = (_plus_2 + _substring_3);
          result = _plus_3;
          SpringRules.LOG.error(("Could not resolve key " + key));
        }
      }
    }
    return result;
  }

  public boolean hasMapping(final MethodDeclaration m) {
    return (((((RuleHelper.isMethodAnnotatedWithName(m, "RequestMapping") || RuleHelper.isMethodAnnotatedWithName(m, "GetMapping")) || 
      RuleHelper.isMethodAnnotatedWithName(m, "PostMapping")) || RuleHelper.isMethodAnnotatedWithName(m, "PutMapping")) || 
      RuleHelper.isMethodAnnotatedWithName(m, "DeleteMapping")) || RuleHelper.isMethodAnnotatedWithName(m, "PatchMapping"));
  }

  public String getMapping(final MethodDeclaration m) {
    final String requestMapping = this.getMappingString(m, "RequestMapping");
    if ((requestMapping != null)) {
      return requestMapping;
    }
    final String getMapping = this.getMappingString(m, "GetMapping");
    if ((getMapping != null)) {
      return getMapping;
    }
    final String postMapping = this.getMappingString(m, "PostMapping");
    if ((postMapping != null)) {
      return postMapping;
    }
    final String putMapping = this.getMappingString(m, "PutMapping");
    if ((putMapping != null)) {
      return putMapping;
    }
    final String deleteMapping = this.getMappingString(m, "DeleteMapping");
    if ((deleteMapping != null)) {
      return deleteMapping;
    }
    final String patchMapping = this.getMappingString(m, "PatchMapping");
    if ((patchMapping != null)) {
      return patchMapping;
    }
    return null;
  }

  public HTTPMethod getHTTPMethod(final MethodDeclaration m) {
    final String requestMapping = this.getMappingString(m, "RequestMapping");
    if ((requestMapping != null)) {
      return HTTPMethod.WILDCARD;
    }
    final String getMapping = this.getMappingString(m, "GetMapping");
    if ((getMapping != null)) {
      return HTTPMethod.GET;
    }
    final String postMapping = this.getMappingString(m, "PostMapping");
    if ((postMapping != null)) {
      return HTTPMethod.POST;
    }
    final String putMapping = this.getMappingString(m, "PutMapping");
    if ((putMapping != null)) {
      return HTTPMethod.PUT;
    }
    final String deleteMapping = this.getMappingString(m, "DeleteMapping");
    if ((deleteMapping != null)) {
      return HTTPMethod.DELETE;
    }
    final String patchMapping = this.getMappingString(m, "PatchMapping");
    if ((patchMapping != null)) {
      return HTTPMethod.PATCH;
    }
    return null;
  }

  public String getMappingString(final MethodDeclaration m, final String annotationName) {
    final String value = RuleHelper.getMethodAnnotationStringValue(m, annotationName);
    if ((value != null)) {
      return value;
    }
    final String path = RuleHelper.getMethodAnnotationStringValue(m, annotationName, "path");
    return path;
  }

  public boolean isRepository(final ITypeBinding binding) {
    return ((((RuleHelper.isImplementingOrExtending(binding, "Repository") || 
      RuleHelper.isImplementingOrExtending(binding, "CrudRepository")) || 
      RuleHelper.isImplementingOrExtending(binding, "JpaRepository")) || 
      RuleHelper.isImplementingOrExtending(binding, "PagingAndSortingRepository")) || 
      RuleHelper.isImplementingOrExtending(binding, "MongoRepository"));
  }

  public boolean isRepository(final CompilationUnit unit) {
    return (((((RuleHelper.isUnitAnnotatedWithName(unit, "Repository") || RuleHelper.isImplementingOrExtending(unit, "Repository")) || 
      RuleHelper.isImplementingOrExtending(unit, "CrudRepository")) || RuleHelper.isImplementingOrExtending(unit, "JpaRepository")) || 
      RuleHelper.isImplementingOrExtending(unit, "PagingAndSortingRepository")) || 
      RuleHelper.isImplementingOrExtending(unit, "MongoRepository"));
  }

  @Override
  public boolean isBuildRule() {
    return false;
  }

  @Override
  public Set<String> getConfigurationKeys() {
    return Set.<String>of();
  }

  @Override
  public String getID() {
    return SpringRules.RULE_ID;
  }

  @Override
  public String getName() {
    return "Spring Rules";
  }

  @Override
  public Set<String> getRequiredServices() {
    return Set.<String>of(SpringRules.JAVA_DISCOVERER_ID, SpringRules.YAML_DISCOVERER_ID, SpringRules.XML_DISCOVERER_ID, SpringRules.PROPERTIES_DISCOVERER_ID, SpringRules.ZUUL_RULE_ID, 
      SpringRules.CLOUD_GATEWAY_RULE_ID);
  }

  @Override
  public Set<String> getDependentServices() {
    return Set.<String>of();
  }
}
