package org.palladiosimulator.retriever.extraction.rules;

import java.nio.file.Path;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.FieldDeclaration;
import org.eclipse.jdt.core.dom.IMethodBinding;
import org.eclipse.jdt.core.dom.ITypeBinding;
import org.eclipse.jdt.core.dom.MethodDeclaration;
import org.eclipse.jdt.core.dom.Type;
import org.palladiosimulator.retriever.extraction.commonalities.CompUnitOrName;
import org.palladiosimulator.retriever.extraction.commonalities.HTTPMethod;
import org.palladiosimulator.retriever.extraction.commonalities.RESTOperationName;
import org.palladiosimulator.retriever.extraction.engine.PCMDetector;
import org.palladiosimulator.retriever.extraction.engine.RuleHelper;
import org.palladiosimulator.retriever.extraction.rules.util.RESTHelper;
import org.palladiosimulator.retriever.services.Rule;
import org.palladiosimulator.retriever.services.blackboard.RetrieverBlackboard;

@SuppressWarnings("all")
public class JaxRSRules implements Rule {
  public static final String RULE_ID = "org.palladiosimulator.retriever.extraction.rules.jax_rs";

  public static final String JAVA_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.java";

  public static final String DEPLOYMENT_RULE_ID = "org.palladiosimulator.retriever.extraction.rules.jax_rs.deployment";

  private static final Map<String, HTTPMethod> SERVLET_METHODS = Map.<String, HTTPMethod>of("doGet", HTTPMethod.GET, "doPost", HTTPMethod.POST, 
    "doDelete", HTTPMethod.DELETE, "doPut", HTTPMethod.PUT, "handleGETRequest", HTTPMethod.GET, "handlePOSTRequest", 
    HTTPMethod.POST, "handleDELETERequest", HTTPMethod.DELETE, "handlePUTRequest", HTTPMethod.PUT);

  @Override
  public void processRules(final RetrieverBlackboard blackboard, final Path path) {
    final CompilationUnit unit = blackboard.<CompilationUnit>getDiscoveredFiles(JaxRSRules.JAVA_DISCOVERER_ID, CompilationUnit.class).get(path);
    if ((unit == null)) {
      return;
    }
    Object _partition = blackboard.getPartition(JaxRSRules.DEPLOYMENT_RULE_ID);
    final Map<Path, String> hostnameMap = ((Map<Path, String>) _partition);
    String hostname = "SERVICE-HOST";
    Path mostSpecificHostnamePath = null;
    Set<Path> _keySet = hostnameMap.keySet();
    for (final Path hostnamePath : _keySet) {
      if ((((hostnamePath != null) && path.startsWith(hostnamePath)) && ((mostSpecificHostnamePath == null) || hostnamePath.startsWith(mostSpecificHostnamePath)))) {
        hostname = hostnameMap.get(hostnamePath);
        mostSpecificHostnamePath = hostnamePath;
      }
    }
    this.processRuleForCompUnit(blackboard, unit, hostname);
  }

  public void processRuleForCompUnit(final RetrieverBlackboard blackboard, final CompilationUnit unit, final String hostname) {
    Object _pCMDetector = blackboard.getPCMDetector();
    final PCMDetector pcmDetector = ((PCMDetector) _pCMDetector);
    if ((pcmDetector == null)) {
      return;
    }
    final CompUnitOrName identifier = new CompUnitOrName(unit);
    final boolean isConverter = RuleHelper.isUnitAnnotatedWithName(unit, "Converter");
    final boolean isUnitController = RuleHelper.isUnitAnnotatedWithName(unit, "Path");
    final boolean isWebServlet = (RuleHelper.isUnitAnnotatedWithName(unit, "WebServlet") || RuleHelper.isImplementingOrExtending(unit, "HttpServlet"));
    boolean _endsWith = identifier.toString().endsWith("Test");
    if (_endsWith) {
      return;
    }
    boolean _isAbstraction = RuleHelper.isAbstraction(unit);
    if (_isAbstraction) {
      return;
    }
    if (isConverter) {
      this.detectDefault(blackboard, unit);
    } else {
      if (isUnitController) {
        String unitPath = RuleHelper.getUnitAnnotationStringValue(unit, "Path");
        if ((unitPath == null)) {
          unitPath = "";
        }
        final String path = ("/" + unitPath);
        pcmDetector.detectComponent(identifier);
        final Consumer<MethodDeclaration> _function = (MethodDeclaration m) -> {
          boolean _isMethodAnnotatedWithName = RuleHelper.isMethodAnnotatedWithName(m, "DELETE", "GET", "HEAD", "PUT", "POST", "OPTIONS");
          if (_isMethodAnnotatedWithName) {
            String methodPath = RuleHelper.getMethodAnnotationStringValue(m, "Path");
            if ((methodPath == null)) {
              methodPath = path;
            } else {
              methodPath = ((path + "/") + methodPath);
            }
            methodPath = RESTHelper.replaceArgumentsWithWildcards(methodPath);
            IMethodBinding _resolveBinding = m.resolveBinding();
            RESTOperationName _rESTOperationName = new RESTOperationName(hostname, methodPath);
            pcmDetector.detectCompositeProvidedOperation(identifier, _resolveBinding, _rESTOperationName);
          }
        };
        RuleHelper.getMethods(unit).forEach(_function);
        final Consumer<FieldDeclaration> _function_1 = (FieldDeclaration f) -> {
          pcmDetector.detectRequiredInterfaceWeakly(identifier, f);
        };
        RuleHelper.getFields(unit).forEach(_function_1);
        pcmDetector.detectPartOfComposite(identifier, RuleHelper.getUnitName(unit));
      } else {
        if (isWebServlet) {
          String unitPath_1 = RuleHelper.getUnitAnnotationStringValue(unit, "WebServlet");
          if ((unitPath_1 == null)) {
            unitPath_1 = "";
          }
          final String path_1 = ("/" + unitPath_1);
          pcmDetector.detectComponent(identifier);
          final Consumer<MethodDeclaration> _function_2 = (MethodDeclaration m) -> {
            boolean _containsKey = JaxRSRules.SERVLET_METHODS.containsKey(m.getName().getIdentifier());
            if (_containsKey) {
              IMethodBinding _resolveBinding = m.resolveBinding();
              Set<HTTPMethod> _of = Set.<HTTPMethod>of(JaxRSRules.SERVLET_METHODS.get(m.getName().getIdentifier()));
              RESTOperationName _rESTOperationName = new RESTOperationName(hostname, path_1, _of);
              pcmDetector.detectProvidedOperation(identifier, _resolveBinding, _rESTOperationName);
            }
          };
          RuleHelper.getMethods(unit).forEach(_function_2);
          final Consumer<FieldDeclaration> _function_3 = (FieldDeclaration f) -> {
            pcmDetector.detectRequiredInterfaceWeakly(identifier, f);
          };
          RuleHelper.getFields(unit).forEach(_function_3);
          pcmDetector.detectPartOfComposite(identifier, RuleHelper.getUnitName(unit));
        } else {
          this.detectDefault(blackboard, unit);
        }
      }
    }
    List<Type> _allAbstractParents = RuleHelper.getAllAbstractParents(unit);
    for (final Type parent : _allAbstractParents) {
      {
        final ITypeBinding parentBinding = parent.resolveBinding();
        pcmDetector.detectProvidedInterfaceWeakly(identifier, parentBinding);
        List<IMethodBinding> _methods = RuleHelper.getMethods(parent);
        for (final IMethodBinding m : _methods) {
          pcmDetector.detectProvidedOperationWeakly(identifier, parentBinding, m);
        }
      }
    }
  }

  public void detectDefault(final RetrieverBlackboard blackboard, final CompilationUnit unit) {
    Object _pCMDetector = blackboard.getPCMDetector();
    final PCMDetector pcmDetector = ((PCMDetector) _pCMDetector);
    final CompUnitOrName identifier = new CompUnitOrName(unit);
    pcmDetector.detectComponent(identifier);
    final Consumer<MethodDeclaration> _function = (MethodDeclaration m) -> {
      pcmDetector.detectProvidedOperationWeakly(identifier, m.resolveBinding());
    };
    RuleHelper.getAllPublicMethods(unit).forEach(_function);
    boolean _endsWith = identifier.name().endsWith("ServiceImpl");
    boolean _not = (!_endsWith);
    if (_not) {
      final Consumer<FieldDeclaration> _function_1 = (FieldDeclaration f) -> {
        pcmDetector.detectRequiredInterfaceWeakly(identifier, f);
      };
      RuleHelper.getFields(unit).forEach(_function_1);
    }
  }

  @Override
  public boolean isBuildRule() {
    return false;
  }

  @Override
  public Set<String> getConfigurationKeys() {
    return Set.<String>of();
  }

  @Override
  public String getID() {
    return JaxRSRules.RULE_ID;
  }

  @Override
  public String getName() {
    return "JAX RS Rules";
  }

  @Override
  public Set<String> getRequiredServices() {
    return Set.<String>of(JaxRSRules.JAVA_DISCOVERER_ID, JaxRSRules.DEPLOYMENT_RULE_ID);
  }

  @Override
  public Set<String> getDependentServices() {
    return Set.<String>of();
  }
}
