package org.palladiosimulator.retriever.vulnerability.core;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.jsoup.Connection;
import org.jsoup.Jsoup;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.AttackerFactory;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.CategorySpecification;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.CVEID;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.CVEVulnerability;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.CWEID;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.impl.AttackSpecificationFactoryImpl;
import org.palladiosimulator.retriever.vulnerability.core.api.IVulnerabilityDatabase;
import org.palladiosimulator.retriever.vulnerability.core.api.VulnerabilityDatabaseException;
import org.palladiosimulator.retriever.vulnerability.core.nvd.CvssV31;
import org.palladiosimulator.retriever.vulnerability.core.nvd.CvssV31Data;
import org.palladiosimulator.retriever.vulnerability.core.nvd.DefCveItem;
import org.palladiosimulator.retriever.vulnerability.core.nvd.NvdResponse;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonSyntaxException;

/**
 * A wrapper for the NIST National Vulnerability Database.
 * 
 * @see <a href="https://nvd.nist.gov/">https://nvd.nist.gov/</a>
 * @author Florian Bossert
 */
public class NistVulnerabilityDatabase implements IVulnerabilityDatabase {
    private static final Logger LOG = Logger.getLogger(NistVulnerabilityDatabase.class);
    private static final String API_ENTRY_POINT = "https://services.nvd.nist.gov/rest/json/cves/2.0/?cveId=";
    private static final String API_KEY_HEADER = "apiKey";
    private static final String API_KEY_ENVIRONMENT_VARIABLE = "NIST_NVD_API_KEY";
    private static final int REQUESTS_PER_MINUTE_DEFAULT = 10;
    private static final int REQUESTS_PER_MINUTE_API_KEY = 100;

    private final Map<String, CVEVulnerability> cache = new HashMap<>();
    private final Map<Integer, CWEID> cweIds = new HashMap<>();
    private final CategorySpecification categorySpecification = AttackerFactory.eINSTANCE.createCategorySpecification();

    private final String apiKey;

    private double requestsPerMinute = 10;

    static {
        LOG.setLevel(Level.INFO);
    }

    public NistVulnerabilityDatabase() {
        this(null);
    }

    public NistVulnerabilityDatabase(String apiKey) {
        if (apiKey == null || apiKey.isBlank()) {
            this.apiKey = System.getenv()
                .get(API_KEY_ENVIRONMENT_VARIABLE);
        } else {
            this.apiKey = apiKey;
        }

        if (this.apiKey == null || this.apiKey.isBlank()) {
            requestsPerMinute = REQUESTS_PER_MINUTE_DEFAULT;
        } else {
            requestsPerMinute = REQUESTS_PER_MINUTE_API_KEY;
        }
    }

    @Override
    public CVEVulnerability getCVEVulnerability(String identifier, List<Integer> cweIdentifiers)
            throws VulnerabilityDatabaseException {
        if (cache.containsKey(identifier)) {
            if (cache.get(identifier) == null) {
                throw new VulnerabilityDatabaseException("NVD API did not return any vulnerabilities!");
            }
            return cache.get(identifier);
        }

        // Sleep in between requests to comply with the API's specifications.
        // Add 50% of sleep time just to be sure to comply.
        try {
            Thread.sleep((long) (1.5 * 60000 / requestsPerMinute));
        } catch (InterruptedException ignored) {
            // If the sleep is interrupted, just go ahead and contact the API.
        }

        String url = API_ENTRY_POINT + identifier;

        String jsonResponse = null;
        try {
            Connection connection = Jsoup.connect(url);
            if (apiKey != null) {
                connection = connection.header(API_KEY_HEADER, apiKey);
            }
            jsonResponse = connection.ignoreContentType(true)
                .ignoreHttpErrors(true) // No vulnerability found -> HTTP 404
                .get()
                .body()
                .text();
        } catch (IOException e) {
            throw new VulnerabilityDatabaseException("Could not contact NVD API!", e);
        }

        Gson gson = new GsonBuilder().setDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS")
            .create();
        NvdResponse response = null;
        try {
            response = gson.fromJson(jsonResponse, NvdResponse.class);
        } catch (JsonSyntaxException e) {
            throw new VulnerabilityDatabaseException(
                    "Could not process NVD API response: " + e + "\nResponse was:\n" + jsonResponse);
        }

        if (response.getVulnerabilities()
            .isEmpty()) {
            cache.put(identifier, null);
            throw new VulnerabilityDatabaseException("NVD API did not return any vulnerabilities!");
        }
        DefCveItem cveItem = response.getVulnerabilities()
            .get(0);
        // The CVE ID directly from the database, to avoid inconsistencies
        String cveId = cveItem.getCve()
            .getId();

        List<CvssV31> metricsV31 = cveItem.getCve()
            .getMetrics()
            .getCvssMetricV31();
        CVEVulnerability vulnerability = null;

        if (metricsV31 != null && !metricsV31.isEmpty()) {
            vulnerability = createCVEVulnFromCVSS(cveId, metricsV31.get(0)
                .getCvssData(), cweIdentifiers);
            LOG.info("Database processed CVSS for " + identifier);
            cache.put(cveId, vulnerability);
            return vulnerability;
        } else {
            cache.put(identifier, null);
            throw new VulnerabilityDatabaseException("Database did not return CVSS for the CVE!");
        }

    }

    @Override
    public CategorySpecification getCategorySpecification() {
        return categorySpecification;
    }

    private CVEVulnerability createCVEVulnFromCVSS(String cveId, CvssV31Data cvss, List<Integer> cweIdentifiers) {
        CVEVulnerability vulnerability = AttackSpecificationFactoryImpl.eINSTANCE.createCVEVulnerability();
        setCveId(vulnerability, cveId);

        for (Integer cweIdentifier : cweIdentifiers) {
            addCweId(vulnerability, cweIdentifier);
        }

        vulnerability.setAttackVector(CvssConverter.convert(cvss.getAttackVector()));
        vulnerability.setPrivileges(CvssConverter.convert(cvss.getPrivilegesRequired()));
        vulnerability.setConfidentialityImpact(CvssConverter.toConfImpact(cvss.getConfidentialityImpact()));
        vulnerability.setIntegrityImpact(CvssConverter.toIntegImpact(cvss.getIntegrityImpact()));
        vulnerability.setAvailabilityImpact(CvssConverter.toAvailImpact(cvss.getAvailabilityImpact()));
        return vulnerability;
    }

    private void setCveId(CVEVulnerability vulnerability, String cveId) {
        CVEID cveIdObj = AttackSpecificationFactoryImpl.eINSTANCE.createCVEID();

        cveIdObj.setEntityName(cveId);
        cveIdObj.setCveID(cveId);

        vulnerability.setEntityName(cveId);

        categorySpecification.getCategories()
            .add(cveIdObj);
        vulnerability.setCveID(cveIdObj);
    }

    private void addCweId(CVEVulnerability vulnerability, Integer cweId) {
        CWEID cweIdObj;

        if (cweIds.containsKey(cweId)) {
            cweIdObj = cweIds.get(cweId);
        } else {
            cweIdObj = AttackSpecificationFactoryImpl.eINSTANCE.createCWEID();
            String name = "CWE-" + cweId.toString();
            cweIdObj.setEntityName(name);
            cweIdObj.setCweID(cweId);

            categorySpecification.getCategories()
                .add(cweIdObj);
            cweIds.put(cweId, cweIdObj);
        }

        vulnerability.getCweID()
            .add(cweIdObj);
    }
}
