/**
 * Opt4J is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * Opt4J is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 * License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with Opt4J. If not, see http://www.gnu.org/licenses/. 
 */

package org.opt4j.common.archive;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.opt4j.core.Individual;
import org.opt4j.core.Objectives;
import org.opt4j.core.optimizer.Archive;

/**
 * This {@link AbstractArchive} provides some common methods for {@link Archive}
 * s. If one or more new {@link Individual}s are added to this {@link Archive},
 * it is assured that all {@link Individual}s in this {@link Archive} are not
 * Pareto-dominated. The methods {@link #add}, {@link #addAll(Collection)}, and
 * {@link #addAll(Individual...)} are implemented such that each dominated
 * Individual is removed from the archive and with the new (unknown)
 * non-dominated {@link Individual}s the method
 * {@link #updateWithNondominated(Collection)} is called. Actual implementations
 * of this class may still refuse or drop some {@link Individual}s. An
 * {@link Archive} can be a {@link BoundedArchive} if it has a bounded size or
 * an {@link UnboundedArchive}, otherwise.
 * 
 * @author helwig, glass, lukasiewycz
 * 
 */
public abstract class AbstractArchive extends Archive {

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.opt4j.core.optimizer.Archive#update(java.util.Set)
	 */
	@Override
	public boolean update(Set<? extends Individual> individuals) {
		List<Individual> candidates = new ArrayList<Individual>(individuals);
		Objectives o1, o2;

		/*
		 * Remove all individuals that are already in the archive and those that
		 * are dominated among the candidates
		 */
		candidates.removeAll(this);
		for (int i = 0; i < candidates.size() - 1; i++) {
			o1 = candidates.get(i).getObjectives();
			for (int j = i + 1; j < candidates.size(); j++) {
				o2 = candidates.get(j).getObjectives();
				if (o2.weaklyDominates(o1)) {
					candidates.remove(i);
					i--;
					break;
				} else if (o1.weaklyDominates(o2)) {
					candidates.remove(j);
					j--;
				}
			}
		}

		/*
		 * Remove those individuals from the archive which are dominated by the
		 * candidates. Remove those individuals from the candidates which are
		 * dominated by the archive.
		 */
		Iterator<Individual> i1, i2;
		for (i1 = candidates.iterator(); i1.hasNext();) {
			o1 = i1.next().getObjectives();
			for (i2 = this.iterator(); i2.hasNext();) {
				o2 = i2.next().getObjectives();
				if (o1.weaklyDominates(o2)) {
					i2.remove();
				} else if (o2.dominates(o1)) {
					i1.remove();
					break;
				}
			}
		}

		return updateWithNondominated(candidates);
	}

	/**
	 * Adds new {@code candidates} which are already checked to be not
	 * Pareto-dominated by any other individual in this {@link Archive}. All
	 * {@link Individual}s in the {@link Archive} which were dominated by the
	 * candidates have already been removed.
	 * 
	 * @param candidates
	 *            the non-dominated individuals which can be added
	 * @return true if one or more candidates are added to the archive
	 */
	protected abstract boolean updateWithNondominated(Collection<Individual> candidates);

}
