/**
 * Opt4J is free software: you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation, either version 3 of the License, or (at your
 * option) any later version.
 * 
 * Opt4J is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 * License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with Opt4J. If not, see http://www.gnu.org/licenses/. 
 */

/**
 * <p>
 * Provides the classes for general (generic) operators.
 * </p>
 * <p>
 * The {@link org.opt4j.core.Genotype} objects are changed within the
 * optimization process in order to find better solutions. The variation is
 * performed by the {@link org.opt4j.core.optimizer.Operator} classes. The
 * framework already contains several operators:
 * <ul>
 * <li>{@link org.opt4j.operator.algebra.Algebra} - Vector-based operator with
 * terms (restricted to {@link org.opt4j.genotype.DoubleGenotype})</li>
 * <li>{@link org.opt4j.operator.copy.Copy} - Copy operator</li>
 * <li>{@link org.opt4j.operator.crossover.Crossover} - Crossover operator that
 * always creates two offspring from two parents</li>
 * <li>{@link org.opt4j.operator.diversity.Diversity} - Diversity operator that
 * determines the differences between two genotypes</li>
 * <li>{@link org.opt4j.operator.mutate.Mutate} - Mutate operator that changes
 * one genotype</li>
 * <li>{@link org.opt4j.operator.neighbor.Neighbor} - Neighbor operator that
 * changes one genotype</li>
 * </ul>
 * Each {@link org.opt4j.core.optimizer.Operator} is parameterized with the corresponding target {@link
 * org.opt4j.core.Genotype}. Adding custom operators, e.g., a new {@link org.opt4j.operator.crossover.Crossover} operator
 * for the {@link org.opt4j.genotype.DoubleGenotype}, is quite simple. The new operator has to be
 * extended from {@link org.opt4j.operator.crossover.Crossover} with the
 * parameter {@link org.opt4j.genotype.DoubleGenotype} or alternatively directly from the
 * {@link org.opt4j.operator.crossover.CrossoverDouble}. The corresponding
 * crossover method has to be implemented. Finally, the operator has to be added
 * with the
 * {@link org.opt4j.operator.crossover.CrossoverModule#addOperator(Class)}
 * method. Therefore, you can extend a custom module from
 * {@link org.opt4j.operator.crossover.CrossoverModule}.
 * </p>
 * <p>
 * The appropriate operator is determined at runtime by the framework by
 * checking the parameter of the operator. Alternatively, the
 * {@link org.opt4j.operator.Apply} annotation can be used to specify a
 * different target {@link org.opt4j.core.Genotype} class.
 * </p>
 * <p>
 * Creating completely new operators is done by extending the {@link org.opt4j.core.optimizer.Operator}
 * interface to the new operator with the specific method. The new operator
 * implementation can be bound directly within a module in the {@code org.opt4j.operator.OperatorModule#configure(com.google.inject.Binder)}
 * method by {@code
 * bind(CustomOperator.class).to(CustomOperatorImplmenentation.class)}. Note
 * that this operator will only handle exactly its target {@link org.opt4j.core.Genotype}
 * classes.
 * </p>
 * <p>
 * A new generic operator should be instantiated from the
 * {@link org.opt4j.operator.AbstractGenericOperator}. See the existing generic
 * operators.
 * </p>
 * <A NAME="custom_operator"><!-- --></A><h2>Integration of a Custom Operator</h2>
 * To add new operators, two tasks have to be performed:
 * <ul>
 * 	<li>The implementation of the custom operator</li>
 * 	<li>The binding of the operator</li>
 * </ul>
 * In the following, a custom {@link org.opt4j.operator.crossover.Crossover} operator for the
 * {@link org.opt4j.genotype.BooleanGenotype} shall be implemented that performs a bitwise or for
 * one offspring and a bitwise and for the other offspring.
 * 
 * <pre>
 * {@code
 * public class MyOperator implements Crossover<Boolean> {
 * 	public Pair<BooleanGenotype> crossover(BooleanGenotype parent1, BooleanGenotype parent2) {
 * 		BooleanGenotype g1 = parent1.newInstance();
 * 		BooleanGenotype g2 = parent1.newInstance();
 * 
 * 		for (int i = 0; i < parent1.size(); i++) {
 * 			g1.add(i, parent1.get(i) || parent2.get(i));
 * 			g2.add(i, parent1.get(i) && parent2.get(i));
 * 		}
 * 
 * 		return new Pair<BooleanGenotype>(g1, g2);
 * 	}
 * }
 * }
 * </pre>
 * To tell the framework to use this operator, implement a
 * {@link org.opt4j.operator.crossover.CrossoverModule} and add your custom operator.
 * 
 * <pre>
 * {@code
 * public class MyOperatorModule extends CrossoverModule {
 * 	protected void config() {
 * 		addOperator(MyOperator.class);
 * 	}
 * }
 * }
 * </pre>
 * 
 */
package org.opt4j.operator;

