/*
 * Copyright (c) 2015, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package org.openjdk.nashorn.internal.runtime;

import java.io.Serializable;
import org.openjdk.nashorn.internal.objects.NativeSymbol;

/**
 * This class represents a unique, non-String Object property key as defined in ECMAScript 6.
 */
public final class Symbol implements Serializable {

    private final String name;

    private static final long serialVersionUID = -2988436597549486913L;

    /**
     * Symbol constructor
     * @param name symbol name
     */
    public Symbol(final String name) {
        this.name = name;
    }

    @Override
    public String toString() {
        return "Symbol(" + name + ")";
    }

    /**
     * Return the symbol's name
     * @return the name
     */
    public final String getName() {
        return name;
    }

    private Object writeReplace() {
        // If this symbol is a globally registered one, replace it with a
        // GlobalSymbol in serialized stream.
        return NativeSymbol.keyFor(null, this) == name ? new GlobalSymbol(name) : this;
    }

    /**
     * Represents a globally registered (with NativeSymbol._for) symbol in the
     * serialized stream. Upon deserialization, it resolves to the globally
     * registered symbol.
     */
    private static class GlobalSymbol implements Serializable {
        private static final long serialVersionUID = 1L;

        private final String name;

        GlobalSymbol(final String name) {
            this.name = name;
        }

        private Object readResolve() {
            return NativeSymbol._for(null, name);
        }
    }
}
