package org.palladiosimulator.retriever.vulnerability.core;

import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.AttackVector;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.AvailabilityImpact;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.ConfidentialityImpact;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.IntegrityImpact;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.Privileges;
import org.palladiosimulator.retriever.vulnerability.core.nvd.CvssV31Data.AttackVectorType;
import org.palladiosimulator.retriever.vulnerability.core.nvd.CvssV31Data.CiaType;
import org.palladiosimulator.retriever.vulnerability.core.nvd.CvssV31Data.PrivilegesRequiredType;

/**
 * Utility class to convert from the NVD API CVSS objects to the attackSpecification CVSS objects.
 * 
 * @see org.palladiosimulator.retriever.vulnerability.core.nvd_api
 * @see org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification
 * 
 * @author Florian Bossert
 */
public final class CvssConverter {
    private CvssConverter() {
        throw new IllegalStateException("This utility class should not be instantiated!");
    }

    /**
     * Converts an NVD {@code CiaType} to a {@code ConfidentialityImpact}.
     * 
     * @param ciaType
     *            the {@code CiaType} to convert
     * @return the converted {@code ConfidentialityImpact}
     */
    public static ConfidentialityImpact toConfImpact(CiaType ciaType) {
        switch (ciaType) {
        case NONE:
            return ConfidentialityImpact.NONE;
        case HIGH:
            return ConfidentialityImpact.HIGH;
        case LOW:
            return ConfidentialityImpact.LOW;
        }
        return null;
    }

    /**
     * Converts an NVD {@code CiaType} to an {@code IntegrityImpact}.
     * 
     * @param ciaType
     *            the {@code CiaType} to convert
     * @return the converted {@code IntegrityImpact}
     */
    public static IntegrityImpact toIntegImpact(CiaType ciaType) {
        switch (ciaType) {
        case NONE:
            return IntegrityImpact.NONE;
        case HIGH:
            return IntegrityImpact.HIGH;
        case LOW:
            return IntegrityImpact.LOW;
        }
        return null;
    }

    /**
     * Converts an NVD {@code CiaType} to an {@code AvailabilityImpact}.
     * 
     * @param ciaType
     *            the {@code CiaType} to convert
     * @return the converted {@code AvailabilityImpact}
     */
    public static AvailabilityImpact toAvailImpact(CiaType ciaType) {
        switch (ciaType) {
        case NONE:
            return AvailabilityImpact.NONE;
        case HIGH:
            return AvailabilityImpact.HIGH;
        case LOW:
            return AvailabilityImpact.LOW;
        }
        return null;
    }

    /**
     * Converts an NVD {@code AttackVectorType} to an {@code AttackVector}.
     * 
     * @param attackVectorType
     *            the {@code AttackVectorType} to convert
     * @return the converted {@code AttackVector}
     */
    public static AttackVector convert(AttackVectorType attackVectorType) {
        switch (attackVectorType) {
        case ADJACENT_NETWORK:
            return AttackVector.ADJACENT_NETWORK;
        case LOCAL:
            return AttackVector.LOCAL;
        case NETWORK:
            return AttackVector.NETWORK;
        case PHYSICAL:
            // Does not exist in the model.
            break;
        }
        return null;
    }

    /**
     * Converts an NVD {@code PrivilegesRequiredType} to a {@code Privileges} object.
     * 
     * @param privilegesRequired
     *            the {@code PrivilegesRequiredType} to convert
     * @return the converted {@code Privileges} object.
     */
    public static Privileges convert(PrivilegesRequiredType privilegesRequired) {
        switch (privilegesRequired) {
        case HIGH:
            return Privileges.SPECIAL;
        case LOW:
            return Privileges.LOW;
        case NONE:
            return Privileges.NONE;
        }
        return null;
    }
}
