package org.palladiosimulator.retriever.vulnerability.core;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Writer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.SimpleLayout;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.xmi.impl.XMIResourceFactoryImpl;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.AttackerSystemSpecificationContainer;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.CategorySpecification;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.VulnerabilityContainer;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.attackSpecification.Vulnerability;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.impl.AttackerFactoryImpl;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.pcmIntegration.PCMElement;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.pcmIntegration.SystemIntegration;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.pcmIntegration.VulnerabilitySystemIntegration;
import org.palladiosimulator.pcm.confidentiality.attackerSpecification.pcmIntegration.impl.PcmIntegrationFactoryImpl;
import org.palladiosimulator.pcm.core.composition.AssemblyContext;
import org.palladiosimulator.pcm.core.entity.Entity;
import org.palladiosimulator.pcm.repository.RepositoryComponent;
import org.palladiosimulator.pcm.system.impl.SystemImpl;
import org.palladiosimulator.retriever.vulnerability.core.api.IStaticCodeAnalysisIssue;
import org.palladiosimulator.retriever.vulnerability.core.api.IStaticCodeAnalysisResult;
import org.palladiosimulator.retriever.vulnerability.core.api.IStaticCodeAnalyst;
import org.palladiosimulator.retriever.vulnerability.core.api.IVulnerabilityDatabase;
import org.palladiosimulator.retriever.vulnerability.core.api.VulnerabilityDatabaseException;

public class SnykCLIStaticCodeAnalyst implements IStaticCodeAnalyst {

    /**
     * The Logger for this analyst.
     */
    private static final Logger LOG = Logger.getLogger(SnykCLIStaticCodeAnalyst.class);

    static {
        LOG.setLevel(Level.DEBUG);
        LOG.addAppender(new ConsoleAppender(new SimpleLayout()));
    }

    /**
     * The vulnerability database for this analyst.
     */
    private final IVulnerabilityDatabase vulnerabilityDatabase;

    /**
     * The path to the snyk executable.
     */
    private Path snykLocation;

    /**
     * The path to the output directory.
     */
    private Path outputLocation;

    /**
     * A token for using the Snyk executable.
     */
    private String snykToken;

    private boolean isSnykExeAuthenticated = false;

    /**
     * Creates a SnykCLIStaticCodeAnalyst which will use the specified Snyk executable.
     *
     * @param snykLocation
     *            a path to a Snyk executable
     * @param outputLocation
     *            a path to the output directory
     * @param apiKey
     *            an API key for the vulnerability database
     * @param snykToken
     *            a token for using the Snyk executable.
     */
    public SnykCLIStaticCodeAnalyst(Path snykLocation, Path outputLocation, String apiKey, String snykToken) {
        this.snykLocation = snykLocation;
        this.outputLocation = outputLocation;
        this.snykToken = snykToken;
        vulnerabilityDatabase = new NistVulnerabilityDatabase(apiKey);
    }

    @Override
    public IStaticCodeAnalysisResult analyze(Map<org.palladiosimulator.pcm.system.System, Path> systemPaths) {
        return analyze(systemPaths, true);
    }

    public IStaticCodeAnalysisResult analyze(Map<org.palladiosimulator.pcm.system.System, Path> systemPaths,
            boolean saveResult) {
        StaticCodeAnalyisResult result = null;

        for (org.palladiosimulator.pcm.system.System system : systemPaths.keySet()) {
            Path path = systemPaths.get(system);
            boolean isMaven = path.getFileName()
                .toString()
                .equals("pom.xml");
            boolean isDocker = path.getFileName()
                .toString()
                .equals("Dockerfile");
            boolean isGradle = path.getFileName()
                .toString()
                .equals("build.gradle");

            if (isMaven || isDocker || isGradle) {
                String output = this.scanWithSnykCLI(path);
                final StaticCodeAnalyisResult thisResult = this.parseSnykCLIOutput(output);
                result = thisResult;
                if (saveResult) {
                    AttackerSystemSpecificationContainer container = AttackerFactoryImpl.eINSTANCE
                        .createAttackerSystemSpecificationContainer();
                    // This is a writable list
                    List<SystemIntegration> systemIntegrations = (List<SystemIntegration>) container
                        .getVulnerabilities();
                    if (system instanceof SystemImpl) {
                        ((SystemImpl) system).getAssemblyContexts__ComposedStructure()
                            .forEach(x -> systemIntegrations.addAll(annotateResultToEntity(x, thisResult)));
                    } else {
                        systemIntegrations.addAll(annotateResultToEntity(system, result));
                    }
                    // Use the name of the parent directory of the pom.xml/Dockerfile as
                    // identifier for the model file
                    String repositoryName = path.getParent()
                        .getFileName()
                        .toString();
                    saveModel(repositoryName, container, vulnerabilityDatabase.getCategorySpecification());
                    saveSnykOutput(repositoryName, output);
                }
            }
        }

        return result;
    }

    /**
     * Annotates a StaticCodeAnalyisResult to an Palladio Entity. Therefore the vulnerabilities of
     * the result are looked up in Snyk's vulnerability database. The found results will be stored
     * in Vulnerability objects that then will be annotated to the specified entity with the help of
     * VulnerabilitySystemIntegrations
     * 
     * @param entity
     *            the vulnerabilities will be annotated to
     * @param result
     *            of a Snyk analysis
     * @return the constructed VulnerabilitySystemIntegration
     */
    private List<VulnerabilitySystemIntegration> annotateResultToEntity(Entity entity, StaticCodeAnalyisResult result) {
        List<VulnerabilitySystemIntegration> sysIntegs = new ArrayList<>();

        for (var issue : result.getIssues()) {
            Vulnerability vul = getVulnerability(issue.getUrl());
            if (vul == null) {
                continue;
            }

            VulnerabilitySystemIntegration sysInteg = PcmIntegrationFactoryImpl.eINSTANCE
                .createVulnerabilitySystemIntegration();

            sysInteg.setVulnerability(vul);
            PCMElement pcmElement = PcmIntegrationFactoryImpl.eINSTANCE.createPCMElement();

            if ((entity instanceof RepositoryComponent)) {
                pcmElement.setBasiccomponent((RepositoryComponent) entity);
            } else if (entity instanceof AssemblyContext) {
                pcmElement.setBasiccomponent(((AssemblyContext) entity).getEncapsulatedComponent__AssemblyContext());
            } else {
                throw new IllegalArgumentException("Please use RepositoryComponents or AssemblyContexts as arguments");
            }

            sysInteg.setPcmelement(pcmElement);
            sysIntegs.add(sysInteg);
        }

        return sysIntegs;
    }

    /**
     * Crawls the specified Snyk web site for the CWE and CVE identifiers of a vulnerability. Then,
     * if the vulnerability has a CWE identifier, it looks up the details of that and returns a
     * CWEVulnerability containing them. Otherwise, a CVE identifier is created and returned.
     * 
     * @param url
     *            the URL of the Snyk web site for a vulnerability
     * @return the details of the vulnerability or {@code null} if no identifier could be found
     */
    private Vulnerability getVulnerability(String url) {
        Document doc = null;
        try {
            doc = Jsoup.connect(url)
                .get();
        } catch (IOException e) {
            LOG.error("Could not get vulnerability definitions from \"" + url + "\"!");
            return null;
        }
        List<String> identifiers = doc.select("a[href]")
            .stream()
            .map(x -> x.ownText())
            .filter(x -> x.startsWith("CVE-"))
            .collect(Collectors.toList());

        List<Integer> cweIdentifiers = doc.select("a[href]")
            .stream()
            .map(x -> x.ownText())
            .filter(x -> x.startsWith("CWE-"))
            .map(x -> x.substring(4))
            .map(x -> {
                try {
                    return Integer.parseInt(x);
                } catch (NumberFormatException e) {
                    return null;
                }
            })
            .filter(x -> x != null)
            .collect(Collectors.toList());

        // If there are multiple identifiers (which there are probably not),
        // try them all until one succeeds.
        for (String identifier : identifiers) {
            try {
                return vulnerabilityDatabase.getCVEVulnerability(identifier, cweIdentifiers);
            } catch (VulnerabilityDatabaseException e) {
                LOG.warn("Database error for \"" + identifier + "\":\n" + e.getMessage());
            }
        }
        return null;
    }

    /**
     * Takes a path and starts the Snyk CLI command for this path. To run the command a process is
     * build and started. The output is read with the help of a BufferedReader. The complete Snyk
     * output will be returned.
     * 
     * @param path
     *            that will be scanned from Snyk
     * @return Snyk output
     */
    private String scanWithSnykCLI(Path path) {
        if (!isSnykExeAuthenticated) {
            authenticateSnykExe();
        }
        if (path.toFile()
            .exists()) {
            String result = runSnykCommand(path.getParent()
                .toFile(), "test", "--all-sub-projects", "--file=" + path);
            if (result == null) {
                return "";
            } else {
                return result;
            }
        } else {
            LOG.error("File does not exist.");
        }

        return "";
    }

    private void authenticateSnykExe() {
        // Do not log the token
        String currentSnykToken = runSnykCommand(false, null, "config", "get", "api");
        if (currentSnykToken != null && !currentSnykToken.isBlank()) {
            LOG.info("Snyk executable is already authenticated");
            return;
        }
        if (snykToken == null || snykToken.isBlank()) {
            LOG.warn("Snyk executable could not be authenticated!\n"
                    + "Authenticate it manually by running \"snyk auth\" or supply an authentication token!\n"
                    + "If the authentication token is supplied via an environment variable, ignore this warning.");
            return;
        }
        if (runSnykCommand("auth", snykToken) != null) {
            isSnykExeAuthenticated = true;
        }
    }

    private String runSnykCommand(String... args) {
        return runSnykCommand(null, args);
    }

    private String runSnykCommand(File path, String... args) {
        return runSnykCommand(true, path, args);
    }

    private String runSnykCommand(boolean logging, File path, String... args) {
        List<String> commandParams = new ArrayList<>();
        commandParams.add(snykLocation.toString());
        commandParams.addAll(List.of(args));
        ProcessBuilder processBuilder = new ProcessBuilder();
        processBuilder.command(commandParams);
        if (path != null) {
            processBuilder.directory(path);
        }

        try {
            Process process = processBuilder.start();
            BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));

            String line;
            StringBuilder sb = new StringBuilder();
            while ((line = reader.readLine()) != null) {
                if (logging) {
                    LOG.debug(line);
                }
                sb.append(line)
                    .append('\n');
            }

            int exitCode = process.waitFor();
            if (logging) {
                LOG.info("\nExited with error code : " + exitCode);
            }
            return sb.toString();

        } catch (IOException | InterruptedException e) {
            LOG.error(e);
        }

        return null;
    }

    // Declaring multiple patterns for RegEx search. All patterns are concatenated
    // to one full pattern that is used to parse the Snyk output.
    private static String packagePattern = "(?<package>[a-zA-Z0-9\\.:@\\-]*)";
    private static String issueNamePattern = "(?<name>[a-z A-Z\\(\\)]*)";
    private static String severityPattern = "\\[(?<severity>[a-z A-Z]*)\\]";
    private static String urlPattern = "\\[(?<url>(?:https?|ftp|file)://[-a-zA-Z0-9+&@#/%?=~_|!:,.;]*[-a-zA-Z0-9+&@#/%=~_|])\\]";
    private static String fullOutputPattern = issueNamePattern + severityPattern + urlPattern + " in " + packagePattern;

    /**
     * Takes Snyk CLI output and parses it. StaticCodeAnalysisResults will be built out of the
     * parsed information.
     * 
     * @param output
     *            of a Snyk CLI process
     * @return StaticCodeAnalyisResult object containing all parsed data
     */
    // Set public for testing
    public StaticCodeAnalyisResult parseSnykCLIOutput(String output) {
        ArrayList<IStaticCodeAnalysisIssue> issues = new ArrayList<>();

        // Possibility to scan for package manager
        String packetManager = null;

        if (output == null || output.isEmpty()) {
            return new StaticCodeAnalyisResult(issues, packetManager);
        }

        // Snyk CLI lists the issues and uses the '\u2717' character as bullet points.
        // The console output is converted to UTF-8 if the regex did not work.
        // The encoding is not reliable. For example:
        // when running the unit tests via surefire, only the first option works,
        // when using the snyk binary on Windows, only the second one does.
        String[] issueStrings = output.split("\\u2717");
        if (issueStrings.length <= 1 /* no occurrence of \u2717 in the default encoding */) {
            issueStrings = new String(output.getBytes(), StandardCharsets.UTF_8).split("\\u2717");
        }

        Pattern pattern = Pattern.compile(fullOutputPattern);

        // Start from index 1 to skip first non-issue String
        for (int i = 1; i < issueStrings.length; i++) {
            Matcher matcher = pattern.matcher(issueStrings[i]);
            if (matcher.find()) {
                SnykIssue issue = new SnykIssue(matcher.group("url"), matcher.group("name"), matcher.group("package"),
                        matcher.group("severity"));
                issues.add(issue);
            }
        }

        return new StaticCodeAnalyisResult(issues, packetManager);
    }

    /**
     * Saving the specified model with a name
     * 
     * @param name
     *            the name to include in the file name
     * @param model
     *            the model to save
     */
    private void saveModel(String name, AttackerSystemSpecificationContainer model, CategorySpecification categories) {

        Resource.Factory.Registry reg = Resource.Factory.Registry.INSTANCE;
        Map<String, Object> m = reg.getExtensionToFactoryMap();
        m.put("vulnerabilitySystemIntegration", new XMIResourceFactoryImpl());

        // Obtain a new resource set
        ResourceSet resSet = new ResourceSetImpl();

        // create a resource
        // Note: this convoluted path conversion is for Windows machines
        Resource resource = resSet
            .createResource(org.eclipse.emf.common.util.URI.createURI(outputLocation.resolve(name + ".attacker")
                .toUri()
                .toString()));

        resource.getContents()
            .add(model);

        // Add vulnerabilities
        VulnerabilityContainer vulnContainer = AttackerFactoryImpl.eINSTANCE.createVulnerabilityContainer();
        for (SystemIntegration integ : model.getVulnerabilities()) {
            VulnerabilitySystemIntegration vulnInteg = (VulnerabilitySystemIntegration) integ;
            if (vulnInteg.getVulnerability() != null) {
                vulnContainer.getVulnerability()
                    .add(vulnInteg.getVulnerability());
            }
        }
        resource.getContents()
            .add(vulnContainer);

        // Add attack categories
        resource.getContents()
            .add(categories);

        // now save the content.
        try {
            resource.save(Collections.emptyMap());
        } catch (IOException e) {
            LOG.error("An error occurred while saving the result:\n" + e.getMessage());
        }
    }

    private void saveSnykOutput(String name, String output) {
        File snykOutputFile = outputLocation.resolve(name + ".log")
            .toFile();
        try (Writer writer = new FileWriter(snykOutputFile)) {
            writer.append(output);
        } catch (IOException e) {
            LOG.error("An error occurred while saving the result:\n" + e.getMessage());
        }
    }

    @Override
    public IStaticCodeAnalysisResult analyze(String path) {
        return parseSnykCLIOutput(path);
    }

}
