package org.palladiosimulator.retriever.vulnerability.core.api;

import java.util.Date;
import java.util.Map;

public interface IStaticCodeAnalysisIssue {

	/**
	 * Returns the issue's id as a String. The id is used to identify the issue for
	 * further use.
	 * 
	 * @return String representing the issue's id.
	 */
	public long getId();

	/**
	 * Get a URL where one may get further information about the issue.
	 * 
	 * @return Issue's information URL.
	 */
	public String getUrl();

	/**
	 * Returns the issue's name. Use it for a human-readable communication.
	 * 
	 * @return Issue's name.
	 */
	public String getTitle();

	/**
	 * Returns an array of String that represent paths to the dependencies that got
	 * an issue.
	 * 
	 * @return Paths to the Dependencies
	 */
	public String[] getPaths();

	/**
	 * Returns the package identifier for the package that got the issue. The
	 * identifier received form the package manager,
	 * 
	 * @return Package identifier.
	 */
	public String getPackage();

	/**
	 * Returns the package version of the package that got the issue.
	 * 
	 * @return Package version.
	 */
	public String getPackageVersion();

	/**
	 * Returns the level of severity of the issue.
	 * 
	 * @return Severity.
	 */
	public String getSeverity();

	/**
	 * Returns the programming language the package that got the issue is written
	 * in.
	 * 
	 * @return Package's programming language.
	 */
	public String getLanguage();

	/**
	 * Returns the date when the issue was first published.
	 * 
	 * @return Issue's publish data
	 */
	public Date getPublicationTime();

	/**
	 * Return the date when the issue was disclosed to the package maintainers.
	 * 
	 * @return Issue's disclosure time,
	 */
	public Date getDisclosureTime();

	/**
	 * Returns a detailed description of the issue.
	 * 
	 * @return Issue description.
	 */
	public String getDescription();

	/**
	 * Returns a map of common categorization identifiers for example CWE. The maps
	 * keys are the categorization providers. The values are the identifiers.
	 * 
	 * @return Issue's categorization identifiers,
	 */
	public Map<String, String> getIdentifiers();

	/**
	 * Returns the Common Vulnerability Scoring System (CVSS).
	 * 
	 * @return Issue's CVSS
	 */
	public String getCVSSv3();

	/**
	 * Returns a Common Vulnerability Scoring System (CVSS) score to measure the
	 * issue.
	 * 
	 * @return Issue's CVSS score
	 */
	public float getCVSSScore();

}
