package org.palladiosimulator.retriever.extraction.rules;

import com.google.common.collect.Iterables;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import org.openjdk.nashorn.api.tree.BinaryTree;
import org.openjdk.nashorn.api.tree.CompilationUnitTree;
import org.openjdk.nashorn.api.tree.ExpressionTree;
import org.openjdk.nashorn.api.tree.FunctionCallTree;
import org.openjdk.nashorn.api.tree.FunctionDeclarationTree;
import org.openjdk.nashorn.api.tree.IdentifierTree;
import org.openjdk.nashorn.api.tree.LiteralTree;
import org.openjdk.nashorn.api.tree.MemberSelectTree;
import org.openjdk.nashorn.api.tree.ObjectLiteralTree;
import org.openjdk.nashorn.api.tree.PropertyTree;
import org.openjdk.nashorn.api.tree.SimpleTreeVisitorES6;
import org.openjdk.nashorn.api.tree.Tree;
import org.openjdk.nashorn.api.tree.VariableTree;
import org.palladiosimulator.retriever.extraction.commonalities.CompUnitOrName;
import org.palladiosimulator.retriever.extraction.commonalities.RESTName;
import org.palladiosimulator.retriever.extraction.commonalities.RESTOperationName;
import org.palladiosimulator.retriever.extraction.engine.PCMDetector;
import org.palladiosimulator.retriever.extraction.rules.data.GatewayRoute;
import org.palladiosimulator.retriever.services.Rule;
import org.palladiosimulator.retriever.services.blackboard.RetrieverBlackboard;

@SuppressWarnings("all")
public class EcmaScriptRules implements Rule {
  public static final String RULE_ID = "org.palladiosimulator.retriever.extraction.rules.ecmascript";

  public static final String ECMASCRIPT_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.ecmascript";

  public static final String HOSTNAMES_ID = "org.palladiosimulator.retriever.extraction.rules.ecmascript.hostnames";

  public static final String GATEWAY_ROUTES_ID = "org.palladiosimulator.retriever.extraction.rules.ecmascript.routes";

  public static final String DONE_ID = "org.palladiosimulator.retriever.extraction.rules.ecmascript.done";

  private static final CompUnitOrName GATEWAY_NAME = new CompUnitOrName("Gateway");

  private static final String START_NONWORD_CHARS = "^[\\W]+";

  private static final String SEPARATOR = ".";

  private static final String URL_KEYWORD = "url";

  private static final String HTTP_KEYWORD = "http";

  private static final String[] HTTP_REQUESTS = { "connect", "delete", "get", "head", "options", "patch", "post", "put", "trace" };

  private static final String VARIABLE_PREFIX = ":";

  private static final String BLANK = "";

  @Override
  public void processRules(final RetrieverBlackboard blackboard, final Path path) {
    boolean _hasPartition = blackboard.hasPartition(EcmaScriptRules.DONE_ID);
    if (_hasPartition) {
      return;
    }
    final Map<Path, CompilationUnitTree> compilationUnits = blackboard.<CompilationUnitTree>getDiscoveredFiles(EcmaScriptRules.ECMASCRIPT_DISCOVERER_ID, CompilationUnitTree.class);
    final CompilationUnitTree compilationUnit = compilationUnits.get(path);
    if (((compilationUnit == null) && (!compilationUnits.isEmpty()))) {
      return;
    }
    Object _partition = blackboard.getPartition(EcmaScriptRules.GATEWAY_ROUTES_ID);
    Map<Path, List<GatewayRoute>> gatewayRouteMap = ((Map<Path, List<GatewayRoute>>) _partition);
    if ((gatewayRouteMap == null)) {
      gatewayRouteMap = Map.<Path, List<GatewayRoute>>of();
    }
    List<GatewayRoute> gatewayRoutes = List.<GatewayRoute>of();
    Path mostSpecificGatewayPath = null;
    Set<Path> _keySet = gatewayRouteMap.keySet();
    for (final Path gatewayPath : _keySet) {
      if ((((gatewayPath != null) && path.startsWith(gatewayPath)) && ((mostSpecificGatewayPath == null) || gatewayPath.startsWith(mostSpecificGatewayPath)))) {
        gatewayRoutes = gatewayRouteMap.get(gatewayPath);
        mostSpecificGatewayPath = gatewayPath;
      }
    }
    Map<Path, String> hostnameMap = Map.<Path, String>of();
    boolean _hasPartition_1 = blackboard.hasPartition(EcmaScriptRules.HOSTNAMES_ID);
    if (_hasPartition_1) {
      Object _partition_1 = blackboard.getPartition(EcmaScriptRules.HOSTNAMES_ID);
      hostnameMap = ((Map<Path, String>) _partition_1);
    }
    String hostname = "API-HOST";
    Path mostSpecificHostnamePath = null;
    Set<Path> _keySet_1 = hostnameMap.keySet();
    for (final Path hostnamePath : _keySet_1) {
      if ((((hostnamePath != null) && path.startsWith(hostnamePath)) && ((mostSpecificHostnamePath == null) || hostnamePath.startsWith(mostSpecificHostnamePath)))) {
        hostname = hostnameMap.get(hostnamePath);
        mostSpecificHostnamePath = hostnamePath;
      }
    }
    Map<String, Set<String>> httpRequests = Map.<String, Set<String>>of();
    Object _pCMDetector = blackboard.getPCMDetector();
    final PCMDetector pcmDetector = ((PCMDetector) _pCMDetector);
    boolean _isEmpty = compilationUnits.isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      httpRequests = this.findAllHttpRequests(blackboard, compilationUnit);
      Set<String> _keySet_2 = httpRequests.keySet();
      for (final String key : _keySet_2) {
        Set<String> _get = httpRequests.get(key);
        for (final String url : _get) {
          {
            final RESTName mappedURL = this.mapURL(hostname, ("/" + url), gatewayRoutes);
            boolean _isPartOf = mappedURL.isPartOf(("/" + hostname));
            boolean _not_1 = (!_isPartOf);
            if (_not_1) {
              pcmDetector.detectCompositeRequiredInterface(EcmaScriptRules.GATEWAY_NAME, mappedURL);
            }
            RESTOperationName _rESTOperationName = new RESTOperationName(hostname, ("/" + url));
            pcmDetector.detectProvidedOperation(EcmaScriptRules.GATEWAY_NAME, null, _rESTOperationName);
          }
        }
      }
    }
    boolean _isEmpty_1 = httpRequests.isEmpty();
    if (_isEmpty_1) {
      for (final GatewayRoute route : gatewayRoutes) {
        {
          String _targetHost = route.getTargetHost();
          final RESTName mappedURL = new RESTName(_targetHost, "/");
          boolean _isPartOf = mappedURL.isPartOf(("/" + hostname));
          boolean _not_1 = (!_isPartOf);
          if (_not_1) {
            pcmDetector.detectCompositeRequiredInterface(EcmaScriptRules.GATEWAY_NAME, mappedURL);
          }
          RESTOperationName _rESTOperationName = new RESTOperationName(hostname, "/");
          pcmDetector.detectProvidedOperation(EcmaScriptRules.GATEWAY_NAME, null, _rESTOperationName);
        }
      }
      blackboard.addPartition(EcmaScriptRules.DONE_ID, Boolean.valueOf(true));
    }
  }

  public Map<String, Set<String>> findAllHttpRequests(final RetrieverBlackboard blackboard, final CompilationUnitTree unit) {
    String _sourceName = unit.getSourceName();
    int _lastIndexOf = unit.getSourceName().lastIndexOf(EcmaScriptRules.SEPARATOR);
    int _plus = (_lastIndexOf + 1);
    final String source = _sourceName.substring(0, _plus);
    final HashMap<String, String> assignments = this.findVariableAssignments(unit);
    final HashMap<String, Collection<String>> requests = EcmaScriptRules.join(this.findFunctionCallsWithUrls(unit), this.findFunctionDeclarationsWithUrls(unit), 
      this.findDirectHttpRequest(unit));
    final Map<String, Set<String>> normalizedRequests = new HashMap<String, Set<String>>();
    Set<String> _keySet = requests.keySet();
    for (final String key : _keySet) {
      {
        HashSet<String> resolvedUrls = new HashSet<String>();
        Collection<String> _get = requests.get(key);
        for (final String url : _get) {
          boolean _startsWith = url.startsWith(EcmaScriptRules.VARIABLE_PREFIX);
          if (_startsWith) {
            boolean _containsKey = assignments.containsKey(url);
            if (_containsKey) {
              resolvedUrls.add(assignments.get(url));
            }
          } else {
            resolvedUrls.add(url);
          }
        }
        HashSet<String> urlsWithWildcards = new HashSet<String>();
        for (final String url_1 : resolvedUrls) {
          {
            final String urlWithoutInteriorParameters = url_1.replaceAll((EcmaScriptRules.VARIABLE_PREFIX + ".+?/"), "*/");
            final String urlWithoutParameters = urlWithoutInteriorParameters.replaceAll((EcmaScriptRules.VARIABLE_PREFIX + ".*"), "*");
            urlsWithWildcards.add(urlWithoutParameters);
          }
        }
        String _replaceAll = key.replaceAll(EcmaScriptRules.START_NONWORD_CHARS, EcmaScriptRules.BLANK);
        String _plus_1 = (source + _replaceAll);
        normalizedRequests.put(_plus_1, urlsWithWildcards);
      }
    }
    return normalizedRequests;
  }

  public Map<String, Set<String>> findDirectHttpRequest(final Tree element) {
    final HashMap<String, Set<String>> calls = new HashMap<String, Set<String>>();
    element.<Void, Void>accept(new SimpleTreeVisitorES6<Void, Void>() {
      @Override
      public Void visitFunctionCall(final FunctionCallTree node, final Void v) {
        final ExpressionTree memberObject = node.getFunctionSelect();
        if (((((memberObject instanceof MemberSelectTree) && Arrays.<String>stream(EcmaScriptRules.HTTP_REQUESTS).filter(((Predicate<String>) (String r) -> {
          return r.equalsIgnoreCase(((MemberSelectTree) memberObject).getIdentifier());
        })).findAny().isPresent()) && (((MemberSelectTree) memberObject).getExpression() instanceof IdentifierTree)) && 
          ((IdentifierTree) ((MemberSelectTree) memberObject).getExpression()).getName().toLowerCase().contains(EcmaScriptRules.HTTP_KEYWORD))) {
          final MemberSelectTree member = ((MemberSelectTree) memberObject);
          ExpressionTree _expression = member.getExpression();
          final IdentifierTree identifier = ((IdentifierTree) _expression);
          String _name = identifier.getName();
          String _plus = (_name + EcmaScriptRules.SEPARATOR);
          String _identifier = member.getIdentifier();
          final String caller = (_plus + _identifier);
          final HashSet<String> urls = EcmaScriptRules.this.findLiteralsInArguments(node.getArguments());
          boolean _isEmpty = urls.isEmpty();
          boolean _not = (!_isEmpty);
          if (_not) {
            boolean _containsKey = calls.containsKey(caller);
            if (_containsKey) {
              calls.get(caller).addAll(urls);
            } else {
              calls.put(caller, urls);
            }
          }
        }
        return super.visitFunctionCall(node, null);
      }
    }, null);
    return calls;
  }

  public HashMap<String, HashSet<String>> findFunctionCallsWithUrls(final Tree element) {
    final HashMap<String, HashSet<String>> calls = new HashMap<String, HashSet<String>>();
    element.<Void, Void>accept(new SimpleTreeVisitorES6<Void, Void>() {
      @Override
      public Void visitFunctionCall(final FunctionCallTree functionCallTree, final Void v) {
        final HashSet<String> urls = new HashSet<String>();
        String literalValue = EcmaScriptRules.BLANK;
        List<? extends ExpressionTree> _arguments = functionCallTree.getArguments();
        for (final ExpressionTree argument : _arguments) {
          urls.addAll(EcmaScriptRules.this.findLiteralsForIdentifier(argument, EcmaScriptRules.URL_KEYWORD));
        }
        boolean _isEmpty = urls.isEmpty();
        if (_isEmpty) {
          return null;
        }
        if (((functionCallTree.getFunctionSelect() instanceof MemberSelectTree) && (((MemberSelectTree) functionCallTree.getFunctionSelect()).getExpression() instanceof FunctionCallTree))) {
          ExpressionTree _functionSelect = functionCallTree.getFunctionSelect();
          ExpressionTree _expression = ((MemberSelectTree) _functionSelect).getExpression();
          boolean _isEmpty_1 = ((FunctionCallTree) _expression).getArguments().isEmpty();
          if (_isEmpty_1) {
            return super.visitFunctionCall(functionCallTree, null);
          }
          ExpressionTree _functionSelect_1 = functionCallTree.getFunctionSelect();
          ExpressionTree _expression_1 = ((MemberSelectTree) _functionSelect_1).getExpression();
          ExpressionTree _get = ((FunctionCallTree) _expression_1).getArguments().get(0);
          if ((_get instanceof LiteralTree)) {
            ExpressionTree _functionSelect_2 = functionCallTree.getFunctionSelect();
            ExpressionTree _expression_2 = ((MemberSelectTree) _functionSelect_2).getExpression();
            ExpressionTree _get_1 = ((FunctionCallTree) _expression_2).getArguments().get(0);
            final LiteralTree literal = ((LiteralTree) _get_1);
            literalValue = String.valueOf(literal.getValue());
          }
        }
        boolean _isBlank = literalValue.isBlank();
        boolean _not = (!_isBlank);
        if (_not) {
          calls.put(literalValue, urls);
        }
        return super.visitFunctionCall(functionCallTree, null);
      }
    }, null);
    return calls;
  }

  public Map<String, Set<String>> findFunctionDeclarationsWithUrls(final Tree element) {
    final HashMap<String, Set<String>> declarations = new HashMap<String, Set<String>>();
    element.<Void, Void>accept(new SimpleTreeVisitorES6<Void, Void>() {
      @Override
      public Void visitFunctionDeclaration(final FunctionDeclarationTree functionDeclaration, final Void v) {
        final HashSet<String> urls = EcmaScriptRules.this.findLiteralsForIdentifier(functionDeclaration, EcmaScriptRules.URL_KEYWORD);
        boolean _isEmpty = urls.isEmpty();
        boolean _not = (!_isEmpty);
        if (_not) {
          boolean _containsKey = declarations.containsKey(functionDeclaration);
          if (_containsKey) {
            declarations.get(functionDeclaration.getName().getName()).addAll(urls);
          } else {
            declarations.put(functionDeclaration.getName().getName(), urls);
          }
        }
        return super.visitFunctionDeclaration(functionDeclaration, null);
      }
    }, null);
    return declarations;
  }

  public String findLiteralInExpression(final ExpressionTree expression) {
    if ((expression != null)) {
      boolean _matched = false;
      if (expression instanceof LiteralTree) {
        _matched=true;
        return String.valueOf(((LiteralTree)expression).getValue());
      }
      if (!_matched) {
        if (expression instanceof IdentifierTree) {
          _matched=true;
          String _valueOf = String.valueOf(((IdentifierTree)expression).getName());
          return (EcmaScriptRules.VARIABLE_PREFIX + _valueOf);
        }
      }
      if (!_matched) {
        if (expression instanceof MemberSelectTree) {
          _matched=true;
          String _valueOf = String.valueOf(((MemberSelectTree)expression).getIdentifier());
          return (EcmaScriptRules.VARIABLE_PREFIX + _valueOf);
        }
      }
      if (!_matched) {
        if (expression instanceof BinaryTree) {
          _matched=true;
          String _findLiteralInExpression = this.findLiteralInExpression(((BinaryTree)expression).getLeftOperand());
          String _findLiteralInExpression_1 = this.findLiteralInExpression(((BinaryTree)expression).getRightOperand());
          return (_findLiteralInExpression + _findLiteralInExpression_1);
        }
      }
    }
    return EcmaScriptRules.BLANK;
  }

  public HashSet<String> findLiteralsForIdentifier(final Tree element, final String identifier) {
    final HashSet<String> literals = new HashSet<String>();
    element.<Void, Void>accept(new SimpleTreeVisitorES6<Void, Void>() {
      @Override
      public Void visitObjectLiteral(final ObjectLiteralTree objectLiteral, final Void v) {
        List<? extends PropertyTree> _properties = objectLiteral.getProperties();
        for (final PropertyTree property : _properties) {
          if (((property.getKey() instanceof IdentifierTree) && 
            identifier.equalsIgnoreCase(((IdentifierTree) property.getKey()).getName()))) {
            literals.add(EcmaScriptRules.this.findLiteralInExpression(property.getValue()));
            return super.visitObjectLiteral(objectLiteral, null);
          }
        }
        return super.visitObjectLiteral(objectLiteral, null);
      }
    }, null);
    return literals;
  }

  public HashSet<String> findLiteralsInArguments(final List<? extends ExpressionTree> arguments) {
    final HashSet<String> urls = new HashSet<String>();
    for (final ExpressionTree argument : arguments) {
      {
        final String url = this.findLiteralInExpression(argument);
        boolean _isBlank = url.isBlank();
        boolean _not = (!_isBlank);
        if (_not) {
          urls.add(url);
        }
      }
    }
    return urls;
  }

  public HashMap<String, String> findVariableAssignments(final Tree element) {
    final HashMap<String, String> assignments = new HashMap<String, String>();
    element.<Void, Void>accept(new SimpleTreeVisitorES6<Void, Void>() {
      @Override
      public Void visitVariable(final VariableTree node, final Void v) {
        final ExpressionTree binding = node.getBinding();
        String _switchResult = null;
        boolean _matched = false;
        if (binding instanceof MemberSelectTree) {
          _matched=true;
          _switchResult = ((MemberSelectTree)binding).getIdentifier();
        }
        if (!_matched) {
          if (binding instanceof IdentifierTree) {
            _matched=true;
            _switchResult = ((IdentifierTree)binding).getName();
          }
        }
        if (!_matched) {
          _switchResult = EcmaScriptRules.BLANK;
        }
        final String id = _switchResult;
        final String url = EcmaScriptRules.this.findLiteralInExpression(node.getInitializer());
        if (((!id.isBlank()) && (!url.isBlank()))) {
          assignments.put((EcmaScriptRules.VARIABLE_PREFIX + id), url);
        }
        return super.visitVariable(node, null);
      }
    }, null);
    return assignments;
  }

  public static HashMap<String, Collection<String>> join(final Map<String, ? extends Set<String>>... maps) {
    final HashMap<String, Collection<String>> join = new HashMap<String, Collection<String>>();
    for (final Map<String, ? extends Set<String>> map : maps) {
      Set<String> _keySet = map.keySet();
      for (final String key : _keySet) {
        boolean _containsKey = join.containsKey(key);
        if (_containsKey) {
          Iterables.<String>addAll(join.get(key), map.get(key));
        } else {
          join.put(key, map.get(key));
        }
      }
    }
    return join;
  }

  public RESTName mapURL(final String host, final String url, final List<GatewayRoute> routes) {
    for (final GatewayRoute route : routes) {
      boolean _matches = route.matches(url);
      if (_matches) {
        return route.applyTo(url);
      }
    }
    return new RESTName(host, url);
  }

  @Override
  public boolean isBuildRule() {
    return false;
  }

  @Override
  public Set<String> getConfigurationKeys() {
    return Set.<String>of();
  }

  @Override
  public String getID() {
    return EcmaScriptRules.RULE_ID;
  }

  @Override
  public String getName() {
    return "ECMAScript Rules";
  }

  @Override
  public Set<String> getRequiredServices() {
    return Set.<String>of(EcmaScriptRules.ECMASCRIPT_DISCOVERER_ID);
  }

  @Override
  public Set<String> getDependentServices() {
    return Set.<String>of();
  }
}
