package org.palladiosimulator.retriever.extraction.rules;

import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.function.Function;
import org.apache.log4j.Logger;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.palladiosimulator.retriever.extraction.rules.data.GatewayRoute;
import org.palladiosimulator.retriever.extraction.rules.util.ProjectHelper;
import org.palladiosimulator.retriever.extraction.rules.util.SpringHelper;
import org.palladiosimulator.retriever.services.Rule;
import org.palladiosimulator.retriever.services.blackboard.RetrieverBlackboard;

@SuppressWarnings("all")
public class SpringGatewayRules implements Rule {
  private static final Logger LOG = Logger.getLogger(SpringGatewayRules.class);

  public static final String RULE_ID = "org.palladiosimulator.retriever.extraction.rules.spring.cloudgateway";

  public static final String YAML_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.yaml";

  public static final String YAML_MAPPERS_KEY = (SpringGatewayRules.YAML_DISCOVERER_ID + ".mappers");

  public static final String XML_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.xml";

  public static final String PROPERTIES_DISCOVERER_ID = "org.palladiosimulator.retriever.extraction.discoverers.properties";

  public static final String ECMASCRIPT_RULE_ID = "org.palladiosimulator.retriever.extraction.rules.ecmascript";

  public static final String ECMASCRIPT_ROUTES_ID = "org.palladiosimulator.retriever.extraction.rules.ecmascript.routes";

  public static final String ECMASCRIPT_HOSTNAMES_ID = "org.palladiosimulator.retriever.extraction.rules.ecmascript.hostnames";

  @Override
  public void processRules(final RetrieverBlackboard blackboard, final Path path) {
    Object _partition = blackboard.getPartition(SpringGatewayRules.YAML_DISCOVERER_ID);
    final Map<Path, Iterable<Map<String, Object>>> rawYamls = ((Map<Path, Iterable<Map<String, Object>>>) _partition);
    Object _partition_1 = blackboard.getPartition(SpringGatewayRules.YAML_MAPPERS_KEY);
    final Map<Path, Function<String, Optional<String>>> yamlMappers = ((Map<Path, Function<String, Optional<String>>>) _partition_1);
    final Map<Path, Properties> propertyFiles = blackboard.<Properties>getDiscoveredFiles(SpringGatewayRules.PROPERTIES_DISCOVERER_ID, Properties.class);
    final Path projectRoot = ProjectHelper.findProjectRoot(path, "pom.xml");
    Map<Path, List<GatewayRoute>> routeMap = new HashMap<Path, List<GatewayRoute>>();
    boolean _hasPartition = blackboard.hasPartition(SpringGatewayRules.RULE_ID);
    if (_hasPartition) {
      Object _partition_2 = blackboard.getPartition(SpringGatewayRules.RULE_ID);
      routeMap = ((Map<Path, List<GatewayRoute>>) _partition_2);
    }
    boolean _containsKey = routeMap.containsKey(projectRoot);
    if (_containsKey) {
      return;
    }
    Function<String, Optional<String>> _xifexpression = null;
    if ((projectRoot == null)) {
      _xifexpression = null;
    } else {
      _xifexpression = yamlMappers.get(
        SpringHelper.findFile(yamlMappers.keySet(), projectRoot.resolve("src/main/resources"), 
          Set.<String>of("bootstrap.yaml", "bootstrap.yml")));
    }
    final Function<String, Optional<String>> bootstrapYaml = _xifexpression;
    Properties _xifexpression_1 = null;
    if ((projectRoot == null)) {
      _xifexpression_1 = null;
    } else {
      _xifexpression_1 = propertyFiles.get(
        SpringHelper.findFile(propertyFiles.keySet(), projectRoot.resolve("src/main/resources"), 
          Set.<String>of("application.properties")));
    }
    final Properties applicationProperties = _xifexpression_1;
    final String applicationName = SpringHelper.getFromYamlOrProperties("spring.application.name", bootstrapYaml, applicationProperties);
    Iterable<Map<String, Object>> _xifexpression_2 = null;
    if ((projectRoot == null)) {
      _xifexpression_2 = null;
    } else {
      _xifexpression_2 = rawYamls.get(
        SpringHelper.findFile(yamlMappers.keySet(), projectRoot.resolve("src/main/resources"), 
          Set.<String>of("application.yaml", "application.yml")));
    }
    final Iterable<Map<String, Object>> rawApplicationYaml = _xifexpression_2;
    final List<GatewayRoute> routes = this.collectRoutes(rawApplicationYaml);
    for (final GatewayRoute route : routes) {
      String _path = route.getPath();
      String _plus = ((("Route in " + applicationName) + ": ") + _path);
      String _plus_1 = (_plus + " -> ");
      String _targetHost = route.getTargetHost();
      String _plus_2 = (_plus_1 + _targetHost);
      SpringGatewayRules.LOG.warn(_plus_2);
    }
    routeMap.put(projectRoot, routes);
    boolean _hasPartition_1 = blackboard.hasPartition(SpringGatewayRules.RULE_ID);
    if (_hasPartition_1) {
      Object _partition_3 = blackboard.getPartition(SpringGatewayRules.RULE_ID);
      ((Map<Path, List<GatewayRoute>>) _partition_3).putAll(routeMap);
    } else {
      blackboard.addPartition(SpringGatewayRules.RULE_ID, routeMap);
    }
    boolean _hasPartition_2 = blackboard.hasPartition(SpringGatewayRules.ECMASCRIPT_ROUTES_ID);
    if (_hasPartition_2) {
      Object _partition_4 = blackboard.getPartition(SpringGatewayRules.ECMASCRIPT_ROUTES_ID);
      final Map<Path, List<GatewayRoute>> ecmaScriptRouteMap = ((Map<Path, List<GatewayRoute>>) _partition_4);
      boolean _containsKey_1 = ecmaScriptRouteMap.containsKey(projectRoot);
      if (_containsKey_1) {
        ecmaScriptRouteMap.get(projectRoot).addAll(routes);
      } else {
        ecmaScriptRouteMap.put(projectRoot, routes);
      }
    } else {
      blackboard.addPartition(SpringGatewayRules.ECMASCRIPT_ROUTES_ID, routeMap);
    }
    if ((applicationName != null)) {
      Map<Path, String> hostnameMap = new HashMap<Path, String>();
      boolean _hasPartition_3 = blackboard.hasPartition(SpringGatewayRules.ECMASCRIPT_HOSTNAMES_ID);
      if (_hasPartition_3) {
        Object _partition_5 = blackboard.getPartition(SpringGatewayRules.ECMASCRIPT_HOSTNAMES_ID);
        hostnameMap = ((Map<Path, String>) _partition_5);
      }
      hostnameMap.put(projectRoot, applicationName);
      boolean _hasPartition_4 = blackboard.hasPartition(SpringGatewayRules.ECMASCRIPT_HOSTNAMES_ID);
      boolean _not = (!_hasPartition_4);
      if (_not) {
        blackboard.addPartition(SpringGatewayRules.ECMASCRIPT_HOSTNAMES_ID, hostnameMap);
      }
    }
  }

  public List<GatewayRoute> collectRoutes(final Iterable<Map<String, Object>> applicationYamlIter) {
    final ArrayList<GatewayRoute> result = new ArrayList<GatewayRoute>();
    if (((applicationYamlIter == null) || IterableExtensions.isEmpty(applicationYamlIter))) {
      return result;
    }
    Map<String, Object> _get = ((Map<String, Object>[])Conversions.unwrapArray(applicationYamlIter, Map.class))[0];
    final Map<String, Object> applicationYaml = ((Map<String, Object>) _get);
    if ((applicationYaml == null)) {
      return result;
    }
    final Object springObject = applicationYaml.get("spring");
    if ((!(springObject instanceof Map))) {
      return result;
    }
    final Map<String, Object> spring = ((Map<String, Object>) springObject);
    final Object cloudObject = spring.get("cloud");
    if ((!(cloudObject instanceof Map))) {
      return result;
    }
    final Map<String, Object> cloud = ((Map<String, Object>) cloudObject);
    final Object gatewayObject = cloud.get("gateway");
    if ((!(gatewayObject instanceof Map))) {
      return result;
    }
    final Map<String, Object> gateway = ((Map<String, Object>) gatewayObject);
    final Object routesObject = gateway.get("routes");
    if ((!(routesObject instanceof List))) {
      return result;
    }
    final List<Map<String, Object>> routes = ((List<Map<String, Object>>) routesObject);
    for (final Map<String, Object> route : routes) {
      {
        final Object uriObject = route.get("uri");
        final Object predicatesObject = route.get("predicates");
        final Object filtersObject = route.get("filters");
        Optional<String> path = Optional.<String>empty();
        if ((((predicatesObject != null) && (predicatesObject instanceof List)) && 
          (((List<Object>) predicatesObject).get(0) instanceof String))) {
          path = this.getPath(((List<String>) predicatesObject));
        }
        int stripPrefixLength = 0;
        if ((((filtersObject != null) && (filtersObject instanceof List)) && 
          (((List<Object>) filtersObject).get(0) instanceof String))) {
          stripPrefixLength = this.getStripPrefixLength(((List<String>) filtersObject));
        }
        final boolean hasUri = ((uriObject != null) && (uriObject instanceof String));
        if ((path.isPresent() && hasUri)) {
          final String uri = ((String) uriObject);
          String _get_1 = path.get();
          String _hostname = this.toHostname(uri);
          GatewayRoute _gatewayRoute = new GatewayRoute(_get_1, _hostname, stripPrefixLength);
          result.add(_gatewayRoute);
        }
      }
    }
    return result;
  }

  public Optional<String> getPath(final List<String> predicates) {
    for (final String predicate : predicates) {
      {
        final String prefix = "Path=";
        boolean _startsWith = predicate.startsWith(prefix);
        if (_startsWith) {
          return Optional.<String>of(predicate.substring(prefix.length()));
        }
      }
    }
    return Optional.<String>empty();
  }

  public int getStripPrefixLength(final List<String> filters) {
    for (final String filter : filters) {
      {
        final String prefix = "StripPrefix=";
        boolean _startsWith = filter.startsWith(prefix);
        if (_startsWith) {
          final String stripPrefixLength = filter.substring(prefix.length());
          try {
            return Integer.parseInt(stripPrefixLength);
          } catch (final Throwable _t) {
            if (_t instanceof NumberFormatException) {
              return 0;
            } else {
              throw Exceptions.sneakyThrow(_t);
            }
          }
        }
      }
    }
    return 0;
  }

  public String toHostname(final String url) {
    int schemaEnd = url.lastIndexOf("://");
    if ((schemaEnd == (-1))) {
      schemaEnd = (-3);
    }
    final int hostnameStart = (schemaEnd + 3);
    int portIndex = url.indexOf(":", hostnameStart);
    if ((portIndex == (-1))) {
      portIndex = url.length();
    }
    int pathIndex = url.indexOf("/", hostnameStart);
    if ((pathIndex == (-1))) {
      pathIndex = url.length();
    }
    final int hostnameEnd = Math.min(portIndex, pathIndex);
    return url.substring(hostnameStart, hostnameEnd);
  }

  @Override
  public boolean isBuildRule() {
    return false;
  }

  @Override
  public Set<String> getConfigurationKeys() {
    return Set.<String>of();
  }

  @Override
  public String getID() {
    return SpringGatewayRules.RULE_ID;
  }

  @Override
  public String getName() {
    return "Spring Cloud Gateway Rules";
  }

  @Override
  public Set<String> getRequiredServices() {
    return Set.<String>of(SpringGatewayRules.YAML_DISCOVERER_ID, SpringGatewayRules.XML_DISCOVERER_ID, SpringGatewayRules.PROPERTIES_DISCOVERER_ID);
  }

  @Override
  public Set<String> getDependentServices() {
    return Set.<String>of(SpringGatewayRules.ECMASCRIPT_RULE_ID);
  }
}
